<?php

namespace Drupal\a12s_maps_sync;

use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\content_translation\ContentTranslationManager;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\taxonomy\Entity\Vocabulary;
use GuzzleHttp\Exception\GuzzleException;

class AutoConfigManager {

  protected const FIELD_GROUP_WRAPPER = 'group_htabs_wrapper';

  protected const MACHINE_NAME_PREFIX = 'maps';

  public const LIBRARIES_MANAGEMENT_ENTITY_REFERENCE = 'entity_reference';
  public const LIBRARIES_MANAGEMENT_LIST = 'list';

  /**
   * @var MapsApi
   */
  protected MapsApi $mapsApi;

  /**
   * @var ContentTranslationManager
   */
  protected ContentTranslationManager $contentTranslationManager;

  /**
   * @var LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * @var LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * @param \Drupal\a12s_maps_sync\MapsApi $mapsApi
   * @param \Drupal\content_translation\ContentTranslationManager $contentTranslationManager
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   */
  public function __construct(MapsApi $mapsApi, ContentTranslationManager $contentTranslationManager, LanguageManagerInterface $languageManager, LoggerChannelFactoryInterface $logger) {
    $this->mapsApi = $mapsApi;
    $this->contentTranslationManager = $contentTranslationManager;
    $this->languageManager = $languageManager;
    $this->logger = $logger->get('a12s_maps_sync');
  }

  /**
   * Process the autoconfiguration for a converter.
   */
  public function processConverter(ConverterInterface $converter) {
    $autoConfig = $converter->getAutoConfig();
    $this->manageAttributeSets($converter, $autoConfig['attribute_sets'], $autoConfig['libraries_management']);
  }

  /**
   * @param ConverterInterface $converter
   * @param array $attributeSets
   * @param string $librariesManagement
   * @param array $attributes
   * @return void
   * @throws EntityStorageException
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function manageAttributeSets(ConverterInterface $converter, array $attributeSets, string $librariesManagement, array $attributes = []) {
    $this->logger->debug("Autoconfig: Manage attribute sets", ['converter' => $converter->id()]);

    $entityType = $converter->getConverterEntityType();
    $bundle = $converter->getConverterBundle();

    $languages = \Drupal::config('a12s_maps_sync.languages_mapping')->get();
    $defaultLanguage = $this->languageManager->getDefaultLanguage();

    // Start with the default language.
    $default = [$defaultLanguage->getId() => $languages[$defaultLanguage->getId()]];
    unset($languages[$defaultLanguage->getId()]);
    $languages = $default + $languages;

    /** @var \Drupal\Core\Language\LanguageInterface $language */
    foreach ($languages as $languageId => $mapsLanguage) {
      $language = $this->languageManager->getLanguage($languageId);

      if (is_null($language)) {
        continue;
      }

      /** @var \Drupal\Core\Entity\Entity\EntityFormDisplay $formDisplay */
      $formDisplay = \Drupal::entityTypeManager()
        ->getStorage('entity_form_display')
        ->load("{$entityType}.{$bundle}.default");

      if (!$formDisplay) {
        $this->logger->debug(
          "Autoconfig: Creating default form display",
          [
            'entity_type' => $entityType,
            'bundle' => $bundle,
          ]
        );

        $formDisplay = EntityFormDisplay::create([
          'targetEntityType' => $entityType,
          'bundle' => $bundle,
          'mode' => 'default',
        ]);
        $formDisplay->save();
      }

      /** @var \Drupal\Core\Entity\Entity\EntityViewDisplay $display */
      $display = \Drupal::entityTypeManager()
        ->getStorage('entity_view_display')
        ->load("{$entityType}.{$bundle}.default");

      if (!$display) {
        $this->logger->debug(
          "Autoconfig: Creating default view display",
          [
            'entity_type' => $entityType,
            'bundle' => $bundle,
          ]
        );

        $display = EntityViewDisplay::create([
          'targetEntityType' => $entityType,
          'bundle' => $bundle,
          'mode' => 'default',
        ]);
        $display->save();
      }

      if ($formDisplay && $display) {
        $formFieldGroups = $formDisplay->getThirdPartySettings('field_group');
        $fieldGroups = $display->getThirdPartySettings('field_group');

        // Create the main field group in the form display if needed.
        if (!empty($formFieldGroups[self::FIELD_GROUP_WRAPPER])) {
          $mainFormFieldGroup = $formFieldGroups[self::FIELD_GROUP_WRAPPER];
        }
        else {
          $mainFormFieldGroup = [
            'label' => 'Wrapper',
            'region' => 'content',
            'weight' => 0,
            'format_type' => 'tabs',
            'format_settings' => [
              'direction' => 'horizontal',
            ],
            'parent_name' => '',
            'children' => [],
          ];
        }

        // Create a field group for each attribute set.
        if (empty($attributeSets)) {
          $attributeSets = [NULL];
        };

        foreach ($attributeSets as $attributeSetCode) {
          try {
            $this->manageAttributeSet($converter, $formDisplay, $formFieldGroups, $display, $fieldGroups, $mainFormFieldGroup, $attributeSetCode, $language, $mapsLanguage, $librariesManagement, $attributes);
          } catch (EntityStorageException|Exception\MapsApiException|GuzzleException|MapsException $e) {
            watchdog_exception('a12s_maps_sync', $e);
          }
        }

        $formDisplay->setThirdPartySetting(
          'field_group',
          self::FIELD_GROUP_WRAPPER,
          $mainFormFieldGroup,
        );

        $formDisplay->setStatus(TRUE);
        $formDisplay->save();

        $display->setStatus(TRUE);
        $display->save();
      }
    }

    $converter->save();
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param \Drupal\Core\Entity\Entity\EntityFormDisplay $formDisplay
   * @param array $formFieldGroups
   * @param \Drupal\Core\Entity\Entity\EntityViewDisplay $display
   * @param array $fieldGroups
   * @param array $mainFormFieldGroup
   * @param string $attributeSetCode
   * @param \Drupal\Core\Language\LanguageInterface $language
   * @param string $mapsLanguage
   *
   * @return void
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \Drupal\a12s_maps_sync\Exception\MapsException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  protected function manageAttributeSet(ConverterInterface $converter, EntityFormDisplay $formDisplay, array $formFieldGroups, EntityViewDisplay $display, array $fieldGroups, array &$mainFormFieldGroup, ?string $attributeSetCode, LanguageInterface $language, string $mapsLanguage, string $librariesManagement, array $attributes): void {
    $this->logger->debug(
      "Autoconfig: Manage attribute set",
      [
        'converter' => $converter->id(),
        'attribute_set' => $attributeSetCode,
        'language' => $language->getId(),
      ],
    );

    $profile = $converter->getProfile();

    if (is_null($attributeSetCode)) {
      // If no attribute set code is defined, we'll check if the converter uses a class as a filter.
      $filters = $converter->getFilters();
      foreach ($filters as $filter) {
        if ($filter->getType() === 'class') {
          $classCode = $filter->getValue();
        }
      }

      if (isset($classCode)) {
        // Find the class id from the API.
        $classes = $this->mapsApi->getConfiguration(
          $profile,
          [
            'type' => 'class',
            'code' => $classCode,
            'id_language' => $mapsLanguage,
          ],
        );

        if (!empty($classes)) {
          $class = reset($classes);

          // From the class id, get the attribute sets.
          $objectClassAttributeSets = $this->mapsApi->getConfiguration(
            $profile,
            [
              'type' => 'object_class_has_attribute_set',
              'id' => $class['id'],
              'id_language' => $mapsLanguage,
            ]
          );

          foreach ($attributes as $attributeId) {
            foreach ($objectClassAttributeSets as $objectClassAttributeSet) {
              foreach ($objectClassAttributeSet['data'] as $attributeSetId) {
                $attributeSetsHasAttributes = $this->mapsApi->getConfiguration(
                  $profile,
                  [
                    'type' => 'attribute_set_has_attribute',
                    'id' => $attributeSetId,
                    'id_language' => $mapsLanguage,
                  ]
                );

                foreach ($attributeSetsHasAttributes as $attributeSetHasAttributes) {
                  if (in_array($attributeId, $attributeSetHasAttributes['data'])) {
                    $_attributeSets = $this->mapsApi->getConfiguration(
                      $profile,
                      [
                        'type' => 'attribute_set',
                        'id' => $attributeSetId,
                        'id_language' => $mapsLanguage,
                      ]
                    );

                    $_attributeSet = reset($_attributeSets);

                    $this->manageAttributeSet($converter, $formDisplay, $formFieldGroups, $display, $fieldGroups, $mainFormFieldGroup, $_attributeSet['code'], $language, $mapsLanguage, $librariesManagement, [$attributeId]);
                    if (($key = array_search($attributeId, $attributes)) !== FALSE) {
                      unset($attributes[$key]);
                    }
                    break 3;
                  }
                }
              }
            }
          }
//          $attributeSet = reset($attributeSets);
//          $attributeSetCode = $attributeSet['code'];
        }
      }
    }

    if (!empty($attributeSetCode)) {
      // Load the attribute set from MaPS.
      $attributeSet = $this->mapsApi->getConfiguration(
        $profile,
        [
          'type' => 'attribute_set',
          'code' => $attributeSetCode,
          'id_language' => $mapsLanguage,
        ],
      );
    }

    // Create the form group if needed.
    if (!empty($attributeSet)) {
      $attributeSet = reset($attributeSet);

      $fieldGroupCode = $attributeSet['code'];
      $fieldGroupLabel = $attributeSet['value'];
      $attributeSetId = $attributeSet['id'];
    }
    else {
      $fieldGroupCode = 'default';
      $fieldGroupLabel = 'Default';
      $attributeSetId = NULL;
    }

    $fieldGroupName = trim('group_htab_' . self::MACHINE_NAME_PREFIX . '_' . strtolower($fieldGroupCode), '_');
    if (empty($formFieldGroups[$fieldGroupName])) {
      $this->logger->debug(
        "Autoconfig: creating field group",
        [
          'converter' => $converter->id(),
          'attribute_set' => $fieldGroupCode,
        ]
      );

      $fieldGroupConfig = [
        'label' => $fieldGroupLabel,
        'region' => 'content',
        'weight' => $attributeSetId ?? 0,
        'parent_name' => '',
        'format_type' => 'tab',
        'format_settings' => [
          'formatter' => 'closed',
        ],
        'children' => [],
      ];

      $formFieldGroups[$fieldGroupName] = $fieldGroupConfig;

      $fieldGroups[$fieldGroupName] = $fieldGroupConfig;
      $fieldGroups[$fieldGroupName]['parent'] = self::FIELD_GROUP_WRAPPER;
      $fieldGroups[$fieldGroupName]['format_type'] = 'html_element';

      $mainFormFieldGroup['children'][] = $fieldGroupName;
    }

    $formFieldGroup = $formFieldGroups[$fieldGroupName];
    $fieldGroup = $fieldGroups[$fieldGroupName];

    // Retrieve the attributes for the current attribute set.
    if ($attributeSetId) {
      $attributeSetHasAttribute = $this->mapsApi->getConfiguration(
        $profile,
        [
          'type' => 'attribute_set_has_attribute',
          'id' => $attributeSetId,
          'id_language' => 1,
        ],
      );

      $attributeSetHasAttribute = reset($attributeSetHasAttribute);
    }

    $seq = 0;
    if (!empty($attributes)) {
      // Just loop through the attributes.
      if (empty($attributeSetHasAttribute)) {
        foreach ($attributes as $attributeId) {
          $this->manageAttribute($converter, $formDisplay, $formFieldGroup, $display, $fieldGroup, $fieldGroupName, (int) $attributeId, $language, $mapsLanguage, $fieldGroupCode, $librariesManagement, NULL);
        }
      }
      else {
        // Loop through the attribute set's attribute to keep the sequences.
        foreach ($attributeSetHasAttribute['data'] as $attributeId) {
          $generate = in_array($attributeId, $attributes);
          $seq++;
          $this->manageAttribute($converter, $formDisplay, $formFieldGroup, $display, $fieldGroup, $fieldGroupName, (int) $attributeId, $language, $mapsLanguage, $fieldGroupCode, $librariesManagement, $seq, $generate);
        }
      }
    }
    else if (!empty($attributeSetHasAttribute)) {
      foreach ($attributeSetHasAttribute['data'] as $attributeId) {
        $seq++;
        $this->manageAttribute($converter, $formDisplay, $formFieldGroup, $display, $fieldGroup, $fieldGroupName, (int) $attributeId, $language, $mapsLanguage, $fieldGroupCode, $librariesManagement, $seq);
      }
    }
    else {
      $this->logger->error(
        "Autoconfig: Cannot retrieve attributes for attribute set",
        [
          'converter' => $converter->id(),
          'attribute_set' => $attributeSetCode,
        ]
      );
    }

    if ($language->isDefault()) {
      $formDisplay->setThirdPartySetting(
        'field_group',
        $fieldGroupName,
        $formFieldGroup,
      );

      $display->setThirdPartySetting(
        'field_group',
        $fieldGroupName,
        $formFieldGroup,
      );
    }
    // Add translations.
    else {
      $formFieldGroup['label'] = $fieldGroupLabel;
      $configOverride = $this->languageManager->getLanguageConfigOverride($language->getId(), "core.entity_form_display.{$converter->getConverterEntityType()}.{$converter->getConverterBundle()}.default");
      $data = $configOverride->get('third_party_settings');
      $data['field_group'][$fieldGroupName] = $formFieldGroup;
      $configOverride->set('third_party_settings', $data);
      $configOverride->save();

      $fieldGroup['label'] = $fieldGroupLabel;
      $configOverride = $this->languageManager->getLanguageConfigOverride($language->getId(), "core.entity_view_display.{$converter->getConverterEntityType()}.{$converter->getConverterBundle()}.default");
      $data = $configOverride->get('third_party_settings');
      $data['field_group'][$fieldGroupName] = $fieldGroup;
      $configOverride->set('third_party_settings', $data);
      $configOverride->save();
    }
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param \Drupal\Core\Entity\Entity\EntityFormDisplay $formDisplay
   * @param array $formFieldGroup
   * @param \Drupal\Core\Entity\Entity\EntityViewDisplay $display
   * @param array $fieldGroup
   * @param string $fieldGroupName
   * @param int $attributeId
   * @param \Drupal\Core\Language\LanguageInterface $language
   * @param int $mapsLanguage
   * @param string $attributeSetCode
   *
   * @return void
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\a12s_maps_sync\Exception\MapsApiException
   * @throws \Drupal\a12s_maps_sync\Exception\MapsException
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function manageAttribute(ConverterInterface $converter, EntityFormDisplay $formDisplay, array &$formFieldGroup, EntityViewDisplay $display, array &$fieldGroup, string $fieldGroupName, int $attributeId, LanguageInterface $language, int $mapsLanguage, ?string $attributeSetCode, string $librariesManagement, ?int $seq, bool $generate = TRUE): void {
    $this->logger->debug(
      "Autoconfig: Manage attribute",
      [
        'converter' => $converter->id(),
        'attribute_set' => $attributeSetCode,
        'attribute' => $attributeId,
      ]
    );

    $entityType = $converter->getConverterEntityType();
    $bundle = $converter->getConverterBundle();

    $attribute = $this->mapsApi->getConfiguration(
      $converter->getProfile(),
      [
        'type' => 'attribute',
        'id' => $attributeId,
        'id_language' => $mapsLanguage,
      ],
    );

    if (!empty($attribute)) {
      $attribute = reset($attribute);

      // Generate the field machine name.
      $fieldName = trim(strtolower(str_replace('-', '_', $attribute['code'])), '-');
      if (strlen($fieldName) > (32 - strlen(self::MACHINE_NAME_PREFIX) - 1)) {
        $fieldName = substr(md5($fieldName), 0, 32 - (strlen(self::MACHINE_NAME_PREFIX) + 1));
      }
      $fieldName = self::MACHINE_NAME_PREFIX . '_' . $fieldName;

      if ($generate) {
        // Check if the attribute is in the deny list.
        $attributesDenyList = $converter->getAutoConfig()['attributes_deny_list'] ?? [];
        if (in_array($attribute['code'], $attributesDenyList)) {
          return;
        }

        // Check if the attribute is already mapped.
        foreach ($converter->getMapping() as $mappingItem) {
          if ($mappingItem->getSource() === $attribute['code'] && $mappingItem->getStatus() !== Mapping::MAPPING_STATUS_AUTO) {
            return;
          }
        }

        // Create the field storage if needed.
        $attributeTypeCode = $attribute['data']['attribute_type_code'] ?? NULL;

        // Management for libraries.
        if ($attributeTypeCode === 'library' && $librariesManagement === self::LIBRARIES_MANAGEMENT_ENTITY_REFERENCE) {
          // Define a new "custom" attribute type code.
          $attributeTypeCode = 'library_reference';
        }

        // We'll have a lot of custom processes if the MaPS attribute is of type
        // object or library, and it's needed for the field configuration.
        // So we start here.
        if (in_array($attributeTypeCode, ['object', 'library_reference'])) {
          $relatedConverter = $this->manageRelatedEntityAttribute($converter, $attribute, $attributeSetCode);
        } else {
          $relatedConverter = NULL;
        }

        // Get the matching field type, settings, ..., depending on the attribute
        // type.
        $fieldType = $this->getFieldTypeFromAttributeType($attributeTypeCode);
        $formType = $this->getFormTypeFromAttributeType($attributeTypeCode);
        $displayType = $this->getDisplayTypeFromAttributeType($attributeTypeCode);
        $storageSettings = $this->getStorageSettingsFromAttributeType($attributeTypeCode, $relatedConverter);
        $storageSettings['field_type'] = $fieldType;
        $settings = $this->getSettingsFromAttributeType($attributeTypeCode, $converter->getProfile(), $attribute['data'], $mapsLanguage, $relatedConverter);

        $fieldStorage = FieldStorageConfig::loadByName($entityType, $fieldName);
        if (empty($fieldStorage)) {
          $fieldStorage = FieldStorageConfig::create(array(
            'field_name' => $fieldName,
            'entity_type' => $entityType,
            'type' => $fieldType,
            'settings' => $storageSettings,
            'translatable' => $attribute['data']['localisable'],
            'cardinality' => $attribute['data']['multiple'] ? -1 : 1,
          ));
          $fieldStorage->save();
        }

        // Create the field.
        $field = FieldConfig::loadByName($entityType, $bundle, $fieldName);
        if (empty($field)) {
          $field = FieldConfig::create([
            'field_name' => $fieldName,
            'entity_type' => $entityType,
            'bundle' => $bundle,
            'label' => $attribute['value'],
            'translatable' => $attribute['data']['localisable'],
            'required' => FALSE,
            'settings' => $settings,
          ]);
          $field->save();

          $handler = match ($fieldType) {
            'integer' => 'integer',
            'float' => 'float',
            default => 'default',
          };

          // Create the mapping.
          $mapping = new Mapping(
            $attribute['code'],
            $fieldName,
            $handler,
          );

          if (!is_null($relatedConverter)) {
            $mapping
              ->setHandler('entity_reference')
              ->setConverters([$relatedConverter]);
          }

          $mapping->setStatus(Mapping::MAPPING_STATUS_AUTO);
          $converter->addMappingItem($mapping);
        } else {
          if ($language->isDefault()) {
            $field->set('label', $attribute['value']);
            $field->save();
          } else {
            // Add the translation...
            $configOverride = $this->languageManager->getLanguageConfigOverride($language->getId(), "field.field.{$converter->getConverterEntityType()}.{$converter->getConverterBundle()}.$fieldName");
            $configOverride->set('label', $attribute['value']);
            $configOverride->save();
          }
        }

        // Special cases for libraries.
        if ($attributeTypeCode === 'library') {
          // We have to retrieve all available values for this library.
          $allowedValues = [];
          $libraries = $this->mapsApi->getLibraries($converter->getProfile(), $attribute['id']);

          if ($language->isDefault()) {
            foreach ($libraries as $library) {
              $allowedValues[$library['code']] = $library['value'][$mapsLanguage];
            }

            $fieldStorage->setSetting('allowed_values', $allowedValues);
            $fieldStorage->save();
          } else {
            foreach ($libraries as $library) {
              $allowedValues[] = [
                'value' => $library['code'],
                'label' => $library['value'][$mapsLanguage],
              ];
            }

            // Add the translation...
            $configOverride = $this->languageManager->getLanguageConfigOverride($language->getId(), "field.storage.{$converter->getConverterEntityType()}.$fieldName");
            $settings = $configOverride->get('settings');
            $settings['allowed_values'] = $allowedValues;
            $configOverride->set('settings', $settings);
            $configOverride->save();
          }
        }

        if (!is_null($seq)) {
          // Add the field to the display and form display.
          $formDisplay->setComponent($fieldName, [
            'region' => 'content',
            'type' => $formType,
            'parent_name' => $fieldGroupName,
          ]);
          $display->setComponent($fieldName, [
            'label' => $attribute['value'],
            'region' => 'content',
            'type' => $displayType,
            'parent_name' => $fieldGroupName,
          ]);
        }
      }

      // Update the sequences.
      if ($formDisplayComponent = $formDisplay->getComponent($fieldName)) {
        $formDisplayComponent['weight'] = $seq;
        $formDisplay->setComponent($fieldName, $formDisplayComponent);
      }

      if ($displayComponent = $display->getComponent($fieldName)) {
        $displayComponent['weight'] = $seq;
        $display->setComponent($fieldName, $displayComponent);
      }

      $formFieldGroup['children'][] = $fieldName;
      $fieldGroup['children'][] = $fieldName;
    }
    else {
      $this->logger->error("Autoconfig: Cannot find attribute in API", ['converter' => $converter->id(), 'attribute' => $attributeId]);
    }
  }

  /**
   * @param string $attributeType
   *
   * @return string
   * @throws \Drupal\a12s_maps_sync\Exception\MapsException
   */
  protected function getFieldTypeFromAttributeType(string $attributeType): string {
    return match($attributeType) {
      'string' => 'string',
      'text' => 'string_long',
      'int' => 'integer',
      'float' => 'float',
      'bool' => 'boolean',
      'object', 'library_reference' => 'entity_reference',
      'library' => 'list_string',
      'date' => 'datetime',
      'html' => 'text_long',
      default => throw new MapsException("Unmanaged $attributeType attribute type"),
    };
  }

  /**
   * @param string $attributeType
   *
   * @return string
   */
  protected function getFormTypeFromAttributeType(string $attributeType): string {
    return match($attributeType) {
      'string' => 'string_textfield',
      'text' => 'string_textarea',
      'int', 'float' => 'number',
      'bool' => 'boolean_checkbox',
      'object', 'library_reference' => 'entity_reference_autocomplete',
      'library' => 'options_select',
      'date' => 'datetime_default',
      'html' => 'text_textarea',
    };
  }

  /**
   * @param string $attributeType
   *
   * @return string
   */
  protected function getDisplayTypeFromAttributeType(string $attributeType): string {
    return match($attributeType) {
      'string' => 'string',
      'text' => 'text_default',
      'int' => 'number_integer',
      'float' => 'number_decimal',
      'bool' => 'boolean',
      'object', 'library_reference' => 'entity_reference_label',
      'library' => 'list_default',
      'date' => 'datetime_default',
      'html' => 'text_default',
    };
  }

  /**
   * @param string $attributeType
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface|null $converter
   *
   * @return array
   */
  protected function getStorageSettingsFromAttributeType(string $attributeType, ?ConverterInterface $converter): array {
    return match($attributeType) {
      'string' => [
        'max_length' => 255,
        'case_sensitive' => FALSE,
        'is_ascii' => FALSE,
      ],
      'text' => [
        'case_sensitive' => FALSE,
      ],
      'int', 'float' => [
        'unsigned' => FALSE,
        'size' => 'normal',
      ],
      'bool' => [
        'on_label' => 'On',
        'off_label' => 'Off',
      ],
      'object', 'library_reference' => [
        'target_type' => $converter->getConverterEntityType(),
      ],
      'date' => [
        'datetime_type' => 'date',
      ],
      default => [],
    };
  }

  /**
   * @param string $attributeType
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface|null $converter
   *
   * @return array
   */
  protected function getSettingsFromAttributeType(string $attributeType, ProfileInterface $profile, array $attributeData, int $mapsLanguage, ?ConverterInterface $converter): array {
    if (in_array($attributeType, ['object', 'library_reference'])) {
      return [
        'handler' => 'default:' . $converter->getConverterEntityType(),
        'handler_settings' => [
          'target_bundles' => [
            $converter->getConverterBundle() => $converter->getConverterBundle(),
          ],
          'sort' => [
            'field' => 'name',
            'direction' => 'asc',
          ],
          'auto_create' => FALSE,
          'auto_create_bundle' => '',
        ],
      ];
    }
    else if (in_array($attributeType, ['int', 'float'])) {
      $return = [
        'prefix' => '',
        'suffix' => '',
      ];

      // Get the unit from the MaPS configuration.
      if (!empty($attributeData['idattribute_unit_type'])) {
        $units = $this->mapsApi->getConfiguration(
          $profile,
          [
            'type' => 'unit',
            'id' => $attributeData['idattribute_unit_type'],
            'id_language' => $mapsLanguage,
          ],
        );

        if (!empty($units)) {
          $unit = reset($units);
          $return['suffix'] = $unit['value'];
        }
      }

      return $return;
    };

    return [];
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter
   * @param array $attribute
   * @param string $attributeSetName
   *
   * @return \Drupal\a12s_maps_sync\Entity\ConverterInterface
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\a12s_maps_sync\Exception\MapsException
   */
  protected function manageRelatedEntityAttribute(ConverterInterface $converter, array $attribute, string $attributeSetCode): ConverterInterface {
    switch ($attribute['data']['attribute_type_code']) {
      case 'object':
        $mapsType = 'object';
        $filterType = 'parent_id';
        $filterValue = $attribute['data']['idobject'];
        $mappingSource = 'LABEL';
        break;

      case 'library':
        $mapsType = 'library';
        $filterType = 'library';
        $filterValue = $attribute['id'];
        $mappingSource = 'value';
        break;

      default:
        throw new MapsException("Cannot manage related entity attribute {$attribute['code']}");
    }

    $vid = trim(strtolower(str_replace('-', '_', $attribute['code'])), '-');
    if (strlen($vid) > (32 - strlen(self::MACHINE_NAME_PREFIX) - 1)) {
      $vid = substr(md5($vid), 0, 32 - (strlen(self::MACHINE_NAME_PREFIX) + 1));
    }
    $vid = self::MACHINE_NAME_PREFIX . '_' . $vid;

    // We need a converter to import related objects.
    // Firstly, check if we already a converter for this, or if the converter already exists.
    $newConverter = Converter::load($vid);

    if (!$newConverter) {
      foreach ($converter->getProfile()->getConverters() as $_converter) {
        if ($_converter->getMapsType() === $mapsType) {
          foreach ($_converter->getFilters() as $filter) {
            if ($filter->getType() === $filterType && $filter->getValue() == $filterValue) {
              // Converter is matching.
              $newConverter = $_converter;
              break 2;
            }
          }
        }
      }
    }

    if ($newConverter === NULL) {
      // We need the current converter to be processed next.
      // So if it does not have a weight, we have to force one.
      if (is_null($converter->getWeight())) {
        $converter->setWeight(50);
        $converter->save();
      }

      // Create the new vocabulary.
      $vocabulary = Vocabulary::load($vid);
      if ($vocabulary === NULL) {
        $vocabulary = Vocabulary::create([
          'vid' => $vid,
          'name' => $attribute['value'],
        ]);
        $vocabulary->save();
      }

      $this->contentTranslationManager->setEnabled('taxonomy_term', $vid, TRUE);

      // Create a new converter.
      $newConverter = new Converter(
        [
          'id' => $vid,
          'label' => $attribute['value'],
          'profile_id' => $converter->getProfile()->id(),
          'parent' => "auto-config--{$converter->id()}--{$attributeSetCode}",
          'gid' => [
            'profile',
            'id',
          ],
          'maps_type' => $mapsType,
          'handler_id' => 'default',
          'entity_type' => 'taxonomy_term',
          'bundle' => $vid,
          'filters' => [[
            'type' => $filterType,
            'value' => $filterValue,
          ]],
          'auto_config' => [],
          'mapping' => [[
            'source' => $mappingSource,
            'target' => 'name',
            'converter' => '',
            'handler' => 'default',
            'entity_type' => '',
            'status' => Mapping::MAPPING_STATUS_AUTO,
          ]],
          'published_statuses' => ['published', 'active'],
          'status_management' => 'unpublish',
          'weight' => $converter->getWeight() - 1,
        ],
        'maps_sync_converter'
      );
      $newConverter->save();
    }

    return $newConverter;
  }

}
