<?php

namespace Drupal\a12s_maps_sync\Commands;

use Consolidation\OutputFormatters\StructuredData\PropertyList;
use Drupal\a12s_maps_sync\AutoConfigManager;
use Drupal\a12s_maps_sync\BatchService;
use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\Profile;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\a12s_maps_sync\Exception\StateQueueItemAlreadyInBatchException;
use Drupal\a12s_maps_sync\State;
use Drupal\a12s_maps_sync\StateBatch;
use Drupal\a12s_maps_sync\StateQueueItem;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Lock\LockBackendInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drush\Exceptions\CommandFailedException;
use Drush\Exceptions\UserAbortException;

/**
 * A Drush commandfile for a12s maps sync.
 */
class A12sMapsSyncCommands extends DrushCommands {

  const BATCH_FLUSH_STATE = 'a12s_maps_sync:batch:flush_state';
  const BATCH_STATUS = 'a12s_maps_sync:batch:status';
  const BATCH_PROCESS = 'a12s_maps_sync:batch:process';
  const BATCH_KILL = 'a12s_maps_sync:batch:kill';
  const ROLLBACK_PROFILE = 'a12s_maps_sync:rollback:profile';
  const ROLLBACK_CONVERTER = 'a12s_maps_sync:rollback:converter';
  const IMPORT_PROFILE = 'a12s_maps_sync:import:profile';
  const IMPORT_CONVERTER = 'a12s_maps_sync:import:converter';
  const RELEASE_LOCK = 'a12s_maps_sync:release_lock';
  const LIST_LOCKS = 'a12s_maps_sync:list_locks';
  const AUTO_CONFIG_CONVERTER = 'a12s_maps_sync:auto_config_converter';
  const AUTO_CONFIG_PROFILE = 'a12s_maps_sync:auto_config_profile';
  const IMPORT_ENTITY = 'a12s_maps_sync:import_entity';
  const IMPORT_OBJECT = 'a12s_maps_sync:import_object';
  const FORCE_PROFILE_REIMPORT = 'a12s_maps_sync:force_profile_reimport';
  const FORCE_CONVERTER_REIMPORT = 'a12s_maps_sync:force_converter_reimport';

  /**
   * @param AutoConfigManager $autoConfigManager
   * @param \Drupal\Core\Lock\LockBackendInterface $drupalLock
   */
  public function __construct(
    protected AutoConfigManager $autoConfigManager,
    protected LockBackendInterface $drupalLock,
  ){
    parent::__construct();
  }

  /**
   * Rollback the given profile.
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  #[CLI\Command(name: self::ROLLBACK_PROFILE, aliases: ['amsrp'])]
  #[CLI\Argument(name: 'profile', description: 'The profile id.')]
  #[CLI\Usage(name: 'drush a12s_maps_sync:rollback:profile my_profile')]
  public function rollbackProfile(string $profile): int {
    if (!$this->io()->confirm(dt("Do you really want to rollback the {$profile} profile?"))) {
      throw new UserAbortException();
    }

    $messenger = \Drupal::messenger();

    // Load the profile.
    try {
      $profile = Profile::load($profile);

      $messenger->addMessage('Rollback profile ' . $profile->label());

      /** @var \Drupal\a12s_maps_sync\Entity\Converter $converter */
      foreach ($profile->getConverters() as $converter) {
        $messenger->addMessage("-- Rollback converter {$converter->label()}");

        $results = $converter->rollback();

        $messenger->addStatus('-- ' . count($results) . ' elements deleted');
      }
    }
    catch (EntityStorageException $e) {
      $messenger->addError("The profile $profile does not exist");
      return self::EXIT_FAILURE;
    }

    $messenger->addStatus('Rollback finished');
    return self::EXIT_SUCCESS;
  }

  /**
   * Rollback the given converter.
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  #[CLI\Command(name: self::ROLLBACK_CONVERTER, aliases: ['amsrc'])]
  #[CLI\Argument(name: 'converter', description: 'The converter id.')]
  #[CLI\Usage(name: 'drush a12s_maps_sync:rollback:converter my_converter')]
  public function rollbackConverter(string $converter): int {
    if (!$this->io()->confirm(dt("Do you really want to rollback the {$converter} converter?"))) {
      throw new UserAbortException();
    }

    $results = [];

    $messenger = \Drupal::messenger();

    // Try to load the converter.
    try {
      $converter = Converter::load($converter);
      $messenger->addMessage('Rollback converter ' . $converter->label());

      $results = $converter->rollback();
    } catch (EntityStorageException $e) {
      $messenger->addError($e->getMessage());
      return self::EXIT_FAILURE;
    }

    if (!empty($results)) {
      $messenger->addStatus('Rollback finished');
      $messenger->addStatus(count($results) . ' elements deleted');
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Import the given profile.
   */
  #[CLI\Command(name: self::IMPORT_PROFILE, aliases: ['amsip'])]
  #[CLI\Argument(name: 'profile', description: 'The profile name.')]
  #[CLI\Option(name: 'limit', description: 'Limit the number of elements processed in each batch set.')]
  #[CLI\Option(name: 'force', description: 'Force the import despite the lock status.')]
  #[CLI\Option(name: 'all', description: 'Force the import of all data.')]
  #[CLI\Option(name: 'batch-no-track', description: 'Use a standard process (without using the batch tracking).')]
  #[CLI\Option(name: 'do-not-process', description: 'Will add the process to the queue, but not process it immediately. Not compatible with the batch-no-track option')]
  #[CLI\Option(name: 'force-queue', description: 'Will add the process to the queue, even if a batch is already set for this profile.')]
  public function importProfile(string $profile, array $options = ['limit' => NULL, 'force' => FALSE, 'all' => FALSE, 'batch-no-track' => FALSE, 'do-not-process' => FALSE, 'force-queue' => FALSE]): int {
    // Load the profile.
    try {
      if ($options['all']) {
        $this->forceProfileReimport($profile);
      }

      $profile = Profile::load($profile);

      if ($options['batch-no-track']) {
        $lockName = $this->getLockKey($profile);
        $config = \Drupal::config('a12s_maps_sync.settings');

        if ((bool) $config->get('ignore_lock') === TRUE || $options['force'] || !\Drupal::state()->get($lockName)) {
          // Set the lock.
          \Drupal::state()->set($lockName, time());

          $batch = BatchService::getProfileImportBatchDefinition(
            $profile,
            $options['limit'] ?? $config->get('import_batch_size'),
            $options['force']
          );

          batch_set($batch);
          drush_backend_batch_process();
        }
        else {
          \Drupal::logger('a12s_maps_sync')->error("The {$profile->getPythonProfileId()} profile is locked. You can remove the lock with the following command: drush amsrl {$profile->getPythonProfileId()}");
        }
      }
      else {
        // Add the operation to our state's queue.
        $state = State::load();

        try {
          $state->addToQueue(new StateQueueItem(
            $profile->id()
          ), $options['force-queue']);
          $state->save();
        }
        catch (StateQueueItemAlreadyInBatchException $e) {
          $this->io()->warning("A batch is already set for this profile. Run with --force-queue to add it anyway to the queue");
        }

        // @todo prettify this...
        $this->io()->note(json_encode(State::load()->toArray()));

        if (!$options['do-not-process']) {
          $this->io()->writeln("Start importing profile {$profile->label()}");
          $this->batchProcess();
        }
      }
    }
    catch (EntityStorageException $e) {
      \Drupal::logger('a12s_maps_sync')->error("The profile $profile does not exist");
      return self::EXIT_FAILURE;
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Import the given converter.
   */
  #[CLI\Command(name: self::IMPORT_CONVERTER, aliases: ['amsic'])]
  #[CLI\Argument(name: 'converter', description: 'The converter name.')]
  #[CLI\Option(name: 'limit', description: 'Limit the number of elements processed in each batch set.')]
  #[CLI\Option(name: 'force', description: 'Force the import despite the lock status.')]
  #[CLI\Option(name: 'all', description: 'Force the import of all data.')]
  #[CLI\Option(name: 'batch-no-track', description: 'Use a standard process (without using the batch tracking).')]
  #[CLI\Option(name: 'do-not-process', description: 'Will add the process to the queue, but not process it immediately.')]
  #[CLI\Option(name: 'force-queue', description: 'Will add the process to the queue, even if a batch is already set for this converter.')]
  public function importConverter(string $converter, array $options = ['limit' => NULL, 'force' => FALSE, 'all' => FALSE, 'batch-no-track' => FALSE, 'do-not-process' => FALSE, 'force-queue' => FALSE]): int {
    if ($options['all']) {
      $this->forceConverterReimport($converter);
    }

    $converter = Converter::load($converter);
    if ($options['batch-no-track']) {
      $profile = $converter->getProfile();
      $lockName = $this->getLockKey($profile);

      $config = \Drupal::config('a12s_maps_sync.settings');

      if ($options['force'] || (bool) $config->get('ignore_lock') === TRUE || !\Drupal::state()->get($lockName)) {
        // Set the lock.
        \Drupal::state()->set($lockName, time());
        $limit = $options['limit'] ?? $config->get('import_batch_size');

        // Try to load the converter.
        try {
          $batch = BatchService::getConverterImportBatchDefinition($converter, $limit, $options['force']);
          batch_set($batch);
          drush_backend_batch_process();
        }
        catch (\Exception $e) {
          \Drupal::logger('a12s_maps_sync')->error($e->getMessage());
          return self::EXIT_FAILURE;
        }
      }
      else {
        $message = "The {$profile->getPythonProfileId()} profile is locked. You can remove the lock with the following command: drush amsrl {$profile->getPythonProfileId()}";
        \Drupal::logger('a12s_maps_sync')->error($message);
        $this->io()->error($message);
        return self::EXIT_FAILURE;
      }
    }
    else {
      // Add the operation to our state's queue.
      $state = State::load();

      try {
        $state->addToQueue(new StateQueueItem(
          $converter->getProfile()->id(),
          $converter->id(),
        ), $options['force-queue']);
        $state->save();
      }
      catch (StateQueueItemAlreadyInBatchException $e) {
        $this->io()->warning("A batch is already set for this converter. Run with --force-queue to add it anyway to the queue");
        return self::EXIT_FAILURE;
      }

      // @todo prettify this...
      $this->io()->note(json_encode(State::load()->toArray()));

      if (!$options['do-not-process']) {
        $this->io()->writeln("Start importing converter {$converter->label()} (profile {$converter->getProfile()->label()})");
        $this->batchProcess();
      }
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Release the lock for a profile.
   */
  #[CLI\Command(name: self::RELEASE_LOCK, aliases: ['amsrl'])]
  #[CLI\Argument(name: 'profile', description: 'The profile name.')]
  public function releaseLock(string $profile): int {
    $profile = Profile::load($profile);

    $this->io()->writeln('Releasing lock for profile ' . $profile->label());
    \Drupal::state()->delete($this->getLockKey($profile));
    $this->io()->success('Lock released');

    return self::EXIT_SUCCESS;
  }

  #[CLI\Command(name: self::LIST_LOCKS, aliases: ['amsll'])]
  public function listLocks(): PropertyList {
    $locks = [];
    $profiles = Profile::loadMultiple();
    foreach ($profiles as $profile) {
      $locks[] = $profile->label() . ': ' . \Drupal::state()->get($this->getLockKey($profile)) ?? 'No lock';
    }

    return new PropertyList($locks);
  }

  /**
   * Process the autoconfiguration for a converter.
   */
  #[CLI\Command(name: self::AUTO_CONFIG_CONVERTER, aliases: ['amsacc'])]
  #[CLI\Argument(name: 'converter', description: 'The converter name.')]
  public function autoConfigConverter(string $converter): int {
    $this->output()->writeln('Auto configuration for converter ' . $converter);

    $converter = Converter::load($converter);

    $autoConfig = $converter->getAutoConfig();
    $attributeSets = $autoConfig['attribute_sets'] ?? [];

    if (empty($attributeSets)) {
      $this->output()->writeln('No auto configuration for this converter');
      return self::EXIT_FAILURE;
    }
    else {
      $this->autoConfigManager->processConverter($converter);
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Process the autoconfiguration for a profile.
   */
  #[CLI\Command(name: self::AUTO_CONFIG_PROFILE, aliases: ['amsacp'])]
  #[CLI\Argument(name: 'profile', description: 'The profile name.')]
  public function autoConfigProfile(string $profile): int {
    // @todo use batch.
    $this->output()->writeln('Auto configuration for profile ' . $profile);

    $profile = Profile::load($profile);

    foreach ($profile->getConverters() as $converter) {
      $autoConfig = $converter->getAutoConfig();
      if (empty($autoConfig['attribute_sets'])) {
        continue;
      }

      $this->output()->writeln('-- Auto configuration for converter ' . $converter->label());
      $this->autoConfigManager->manageAttributeSets($converter);
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Import an entity.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\a12s_maps_sync\Exception\MapsException
   */
  #[CLI\Command(name: self::IMPORT_ENTITY, aliases: ['amsie'])]
  #[CLI\Argument(name: 'entityType', description: 'The entity type.')]
  #[CLI\Argument(name: 'entityId', description: 'The entity id.')]
  #[CLI\Option(name: 'profile', description: 'The profile to use.')]
  #[CLI\Option(name: 'converter', description: 'The converter to use.')]
  #[CLI\Option(name: 'with-dependencies', description: 'Whether we want to import all dependencies.')]
  public function importEntity(string $entityType, int $entityId, array $options = ['profile' => NULL, 'converter' => NULL, 'with-dependencies' => FALSE]): int {
    // Load the entity.
    $entity = \Drupal::entityTypeManager()->getStorage($entityType)->load($entityId);

    if (!$entity) {
      throw new MapsException("No entity found ($entityType $entityId)");
    }

    try {
      $profile = !is_null($options['profile']) ? Profile::load($options['profile']) : NULL;
      $converter = !is_null($options['converter']) ? Converter::load($options['converter']) : NULL;

      $batch = BatchService::getEntityImportBatchDefinition($entity, $options['with-dependencies'], $profile, $converter);

      batch_set($batch);
      drush_backend_batch_process();
    } catch (\Exception $e) {
      \Drupal::logger('a12s_maps_sync')->error($e->getMessage());
      return self::EXIT_FAILURE;
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Import a MaPS object.
   */
  #[CLI\Command(name: self::IMPORT_OBJECT, aliases: ['amsio'])]
  #[CLI\Argument(name: 'objectId', description: 'The MaPS id.')]
  #[CLI\Argument(name: 'converterId', description: 'The converter id.')]
  #[CLI\Option(name: 'with-dependencies', description: 'Whether we want to import all dependencies.')]
  public function importObject(int $objectId, string $converterId, array $options = ['with-dependencies' => FALSE]): int {
    try {
      $converter = Converter::load($converterId);
      $batch = BatchService::getObjectImportBatchDefinition($objectId, $converter, $options['with-dependencies']);

      if ($batch) {
        batch_set($batch);
        drush_backend_batch_process();
      }
    } catch (\Exception $e) {
      \Drupal::logger('a12s_maps_sync')->error($e->getMessage());
      return self::EXIT_FAILURE;
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Force the profile reimport.
   *
   * @throws \Drush\Exceptions\CommandFailedException
   */
  #[CLI\Command(name: self::FORCE_PROFILE_REIMPORT, aliases: ['amsfpr'])]
  #[CLI\Argument(name: 'profile', description: 'The profile id.')]
  public function forceProfileReimport(string $profile): int {
    $state = State::load();
    $stateBatch = $state->getBatch();
    if (!is_null($stateBatch) && !is_null($stateBatch->getProfile()) && $stateBatch->getProfile() === $profile) {
      throw new CommandFailedException("Cannot force reimport since there is already a batch set for profile {$profile}");
    }

    $profile = Profile::load($profile);
    if ($profile) {
      $this->output()->writeln('Force reimport for profile ' . $profile->label());

      foreach ($profile->getConverters() as $converter) {
        $this->output()->writeln(' - Force reimport for converter ' . $converter->label());
        $converter->resetLastImportedTime();
      }
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Force the converter reimport.
   *
   * @throws \Drush\Exceptions\CommandFailedException
   */
  #[CLI\Command(name: self::FORCE_CONVERTER_REIMPORT, aliases: ['amsfcr'])]
  #[CLI\Argument(name: 'converter', description: 'The converter id.')]
  public function forceConverterReimport(string $converter): int {
    $converter = Converter::load($converter);

    $state = State::load();
    $stateBatch = $state->getBatch();
    if (!is_null($stateBatch) && !is_null($stateBatch->getProfile()) && $stateBatch->getProfile() === $converter->getProfile()->id()) {
      throw new CommandFailedException("Cannot force reimport since there is already a batch set for profile {$converter->getProfile()->id()}");
    }

    if ($converter) {
      $this->output()->writeln('Force reimport for converter ' . $converter->label());
      $converter->resetLastImportedTime();
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Show the current state.
   */
  #[CLI\Command(name: self::BATCH_STATUS, aliases: ['amsbs'])]
  #[CLI\Usage(name: 'drush a12s_maps_sync:batch:status')]
  public function batchStatus(): PropertyList|string {
    if ($this->isCronRunning()) {
      $this->io()->warning("The Drupal cron is currently running");
    }

    $data = State::load()->toArray();
    return !empty($data) ? new PropertyList($data) : 'No current state';
  }

  /**
   * Flush the current state.
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  #[CLI\Command(name: self::BATCH_FLUSH_STATE, aliases: ['amsbfs'])]
  #[CLI\Usage(name: 'drush a12s_maps_sync:batch:flush_state')]
  public function batchFlushState(): int {
    if (!$this->io()->confirm(dt('Do you want to flush the current state?'))) {
      throw new UserAbortException();
    }

    (new State())->save();

    return self::EXIT_SUCCESS;
  }

  /**
   * Kill the given batch.
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  #[CLI\Command(name: self::BATCH_KILL, aliases: ['amsbk'])]
  #[CLI\Argument(name: 'batchId', description: 'The batch id.')]
  public function batchKill(int $batchId): int {
    if (!$this->io()->confirm(dt("Do you want to kill the batch with id {$batchId}?"))) {
      throw new UserAbortException();
    }

    \Drupal::database()->delete('batch')
      ->condition('bid', $batchId)
      ->execute();

    \Drupal::database()->delete('queue')
      ->condition('name', "drupal_batch:{$batchId}:%", 'LIKE')
      ->execute();

    return self::EXIT_SUCCESS;
  }

  /**
   * Process the batch.
   *
   * @param array $options
   *
   * @return int
   * @throws \Drush\Exceptions\CommandFailedException
   */
  #[CLI\Command(name: self::BATCH_PROCESS, aliases: ['amsbp'])]
  #[CLI\Usage(name: 'drush a12s_maps_sync:batch:process')]
  #[CLI\Option(name: 'limit', description: 'The limit of elements to process.')]
  public function batchProcess(array $options = ['limit' => NULL]): int {
    if ($this->isCronRunning()) {
      throw new CommandFailedException("The Drupal cron is already running.");
    }

    $state = State::load();
    $stateBatch = $state->getBatch();

    if (!is_null($stateBatch)) {
      $profile = Profile::load($stateBatch->getProfile());
      if ($this->isLocked($profile)) {
        throw new CommandFailedException("The profile {$profile->id()} is locked.");
      }

      // Get the first (Drupal) queue item to process.
      $itemIds = $stateBatch->getRemainingItems();

      if (empty($itemIds)) {
        // Import is finished?
        $state->unsetBatch();
        $state->save();
        $this->batchProcess($options);
      }
      else {
        $itemId = reset($itemIds);

        // If the item was already processing, increment the counter.
        if ($itemId == $stateBatch->getItemId()) {
          $stateBatch->incrementItemIterationCount();
        }

        $config = \Drupal::config('a12s_maps_sync.settings');
        $maxIterationsCount = (int) $config->get('batch_max_iterations_count') ?? 4;

        // Check if we need to retry or not.
        if ($stateBatch->getItemIterationCount() > $maxIterationsCount) {
          $errorMessage = "The process for profile {$stateBatch->getProfile()}";
          if ($stateBatch->getConverter()) {
            $errorMessage .= " (converter {$stateBatch->getConverter()})";
          }
          $errorMessage .= "has been launched {$stateBatch->getItemIterationCount()} times. It has been removed from the state.";

          \Drupal::logger('a12s_maps_sync')->error($errorMessage);
          $state->unsetBatch();
          $state->save();

          return self::EXIT_FAILURE;
        }

        $stateBatch
          ->updateUpdated()
          ->setItemId($itemId)
          ->resetItemIterationCount();

        $state->setBatch($stateBatch);
        $state->save();

        drush_batch_command($stateBatch->getBatchId());

        if ($stateBatch->getRemainingItemsCount() === 0) {
          $state->unsetBatch();
          $state->save();
        }
      }
    }
    else {
      $limit = $options['limit'];
      if (is_null($limit)) {
        $config = \Drupal::config('a12s_maps_sync.settings');
        $limit = $config->get('import_batch_size');
      }

      // Get the queue.
      $queue = $state->getQueue();
      if (empty($queue)) {
        $this->io()->note("The queue is empty. Run the a12s_maps_sync:import profile or a12s_maps_sync:import:converter before.");
        return self::EXIT_FAILURE;
      }

      $queueItem = reset($queue);
      $profile = Profile::load($queueItem->getProfile());
      $converterId = $queueItem->getConverter();

      // Construct the batch.
      if (is_null($converterId)) {
        $batch = BatchService::getProfileImportBatchDefinition($profile, $limit);
      }
      else {
        $converter = Converter::load($converterId);
        $batch = BatchService::getConverterImportBatchDefinition($converter, $limit, FALSE);
      }

      // Create the batch.
      batch_set($batch);

      $batch = &batch_get();
      $process_info = [
        'current_set' => 0,
      ];
      $batch += $process_info;

      \Drupal::moduleHandler()->alter('batch', $batch);

      $batchStorage = \Drupal::service('batch.storage');
      $batch['id'] = $batchStorage->getId();

      if (!$batch['id']) {
        throw new CommandFailedException("Cannot generate the batch.");
      }

      $batch['progressive'] = TRUE;

      // Move operations to a job queue. Non-progressive batches will use a
      // memory-based queue.
      foreach ($batch['sets'] as $key => $batch_set) {
        _batch_populate_queue($batch, $key);
      }
      // Store the batch.
      $batchStorage->create($batch);

      // Store the batch in our state.
      $state->setBatch(new StateBatch($batch['id'], $profile->id(), $converterId));
      $state->removeFromQueue($queueItem);
      $state->save();

      $this->batchProcess($options);
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Checks if the Drupal cron is running.
   *
   * @return bool
   */
  private function isCronRunning(): bool {
    return \Drupal::database()->select('semaphore')
      ->condition('name', 'cron')
      ->countQuery()
      ->execute()
      ->fetchField() > 0;
  }

  /**
   * Check if a profile is locked.
   *
   * @param \Drupal\a12s_maps_sync\Entity\Profile $profile
   *
   * @return bool
   */
  private function isLocked(Profile $profile): bool {
    return !empty(\Drupal::state()->get($this->getLockKey($profile)));
  }

  /**
   * Get the lock key for a given profile.
   *
   * @param \Drupal\a12s_maps_sync\Entity\Profile $profile
   *
   * @return string
   */
  private function getLockKey(Profile $profile): string {
    return 'a12s_maps_sync:lock:profile:' . $profile->getPythonProfileId();
  }

}
