<?php

namespace Drupal\a12s_maps_sync\Entity;

use Drupal\a12s_maps_sync\Converter\Filter;
use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Event\ConverterItemImportEvent;
use Drupal\a12s_maps_sync\Exception\MapsApiException;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerInterface;
use Drupal\a12s_maps_sync\Plugin\SourceHandlerInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Language\LanguageManager;
use Drupal\language\Entity\ContentLanguageSettings;

/**
 * Defines the Maps sync converter entity.
 *
 * @ConfigEntityType(
 *   id = "maps_sync_converter",
 *   label = @Translation("Converter"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\a12s_maps_sync\ConverterListBuilder",
 *     "form" = {
 *       "add" = "Drupal\a12s_maps_sync\Form\ConverterForm",
 *       "edit" = "Drupal\a12s_maps_sync\Form\ConverterForm",
 *       "delete" = "Drupal\a12s_maps_sync\Form\ConverterDeleteForm",
 *       "filters" = "Drupal\a12s_maps_sync\Form\ConverterFiltersForm",
 *       "mapping" = "Drupal\a12s_maps_sync\Form\ConverterMappingForm",
 *       "import" = "Drupal\a12s_maps_sync\Form\ConverterImportForm",
 *       "auto_config" = "Drupal\a12s_maps_sync\Form\ConverterAutoConfigForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\a12s_maps_sync\Routing\ConverterHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "maps_sync_converter",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "uuid",
 *     "langcode",
 *     "status",
 *     "dependencies",
 *     "id",
 *     "label",
 *     "profile_id",
 *     "entity_type",
 *     "bundle",
 *     "filters",
 *     "mapping",
 *     "auto_config",
 *     "published_statuses",
 *     "unpublished_statuses",
 *     "deleted_statuses",
 *     "status_management",
 *     "status_property",
 *     "status_property_name",
 *     "gid",
 *     "maps_type",
 *     "handler_id",
 *     "parent",
 *     "weight",
 *     "media_status_published_value",
 *     "status_property"
 *   },
 *   links = {
 *     "canonical" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/edit",
 *     "add-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/add",
 *     "edit-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/edit",
 *     "filters-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/filters",
 *     "mapping-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/mapping",
 *     "auto-config-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/auto-config",
 *     "import-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/import",
 *     "delete-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/delete",
 *     "collection" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter"
 *   }
 * )
 */
class Converter extends ConfigEntityBase implements ConverterInterface {

  /**
   * The Maps sync converter ID.
   *
   * @var string
   */
  protected string $id;

  /**
   * The Maps sync converter label.
   *
   * @var string
   */
  protected string $label;

  /**
   * The filters.
   *
   * @var Filter[]
   */
  protected array $filters = [];

  /**
   * The mapping.
   *
   * @var Mapping[]
   */
  protected ?array $mapping = [];

  /**
   * The automatic configuration.
   *
   * @var array|null
   */
  protected ?array $auto_config = [];

  /**
   * {@inheritdoc}
   */
  public function getProfile(): ?ProfileInterface {
    if ($this->get('profile_id') === NULL) {
      return NULL;
    }
    return Profile::load($this->get('profile_id'));
  }

  /**
   * {@inheritdoc}
   */
  public function setProfileId(string $profile_id): ConverterInterface {
    $this->set('profile_id', $profile_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConverterEntityType(): ?string {
    return $this->get('entity_type');
  }

  /**
   * {@inheritdoc}
   */
  public function setConverterEntityType(string $entity_type): ConverterInterface {
    $this->set('entity_type', $entity_type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConverterBundle(): ?string {
    return $this->get('bundle');
  }

  /**
   * {@inheritdoc}
   */
  public function setConverterBundle(string $bundle): ConverterInterface {
    $this->set('bundle', $bundle);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFilters(): ?array {
    $filters = [];
    foreach ($this->get('filters') as $filter) {
      if (!empty($filter['type']) && (!empty($filter['value'])) || !empty($filter['filtering_type']) && $filter['filtering_type'] !== 'value') {
        $filteringType = !empty($filter['filtering_type']) ? $filter['filtering_type'] : 'value';
        $filters[] = new Filter($filter['type'], $filter['value'] ?? NULL, $filteringType);
      }
    }
    return $filters;
  }

  /**
   * {@inheritdoc}
   */
  public function setFilters(array $filters = []): ConverterInterface {
    $this->set('filters', $filters);
    return $this;
  }

  /**
   * Returns the filters in an associative array format.
   *
   * @return array
   */
  public function getFiltersArray(): array {
    $return = [];
    foreach ($this->getFilters() as $filter) {
      $return[$filter->getType()] = [
        'filtering_type' => $filter->getFilteringType(),
        'value' => $filter->getValue(),
      ];
    }
    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function getMapping(): array {
    $mapping = [];
    foreach ($this->get('mapping') as $mappingItem) {
      $mapping[] = new Mapping(
        $mappingItem['source'],
        $mappingItem['target'],
        $mappingItem['handler'] ?? 'default',
        !empty($mappingItem['append']) ? $mappingItem['append'] : FALSE,
        !empty($mappingItem['status']) ? $mappingItem['status'] : Mapping::MAPPING_STATUS_MANUAL,
        !empty($mappingItem['required']) ? $mappingItem['required'] : FALSE,
        !empty($mappingItem['requiredBehavior']) ? $mappingItem['requiredBehavior'] : NULL,
        $mappingItem['options'] ?? [],
      );
    }

    return !empty($mapping) ? $mapping : [];
  }

  /**
   * {@inheritdoc}
   */
  public function setMapping(array $mapping = []): ConverterInterface {
    $this->set('mapping', $mapping);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAutoConfig(): ?array {
    return $this->get('auto_config');
  }

  /**
   * {@inheritdoc}
   */
  public function setAutoConfig(?array $auto_config): Converter {
    $this->set('auto_config', $auto_config);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPublishedStatuses(): array {
    return $this->get('published_statuses') ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function setPublishedStatuses(array $statuses): ConverterInterface {
    $this->set('published_statuses', $statuses);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getUnpublishedStatuses(): array {
    return $this->get('unpublished_statuses') ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function setUnpublishedStatuses(array $statuses): ConverterInterface {
    $this->set('unpublished_statuses', $statuses);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDeletedStatuses(): array {
    return $this->get('deleted_statuses') ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function setDeletedStatuses(array $statuses): ConverterInterface {
    $this->set('deleted_statuses', $statuses);
    return $this;
  }


  /**
   * {@inheritdoc}
   */
  public function getStatusManagement(): ?string {
    return $this->get('status_management');
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusProperty(): ?string {
    return $this->get('status_property');
  }

  /**
   * {@inheritdoc}
   */
  public function setStatusProperty(string $mediaStatusProperty): ConverterInterface {
    $this->set('status_property', $mediaStatusProperty);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusPropertyName(): ?string {
    return $this->get('status_property_name');
  }

  /**
   * {@inheritdoc}
   */
  public function setStatusPropertyName(string $statusPropertyName): ConverterInterface {
    $this->set('status_property_name', $statusPropertyName);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getMediaStatusPublishedValue(): ?string {
    return $this->get('media_status_published_value');
  }

  /**
   * @inheritDoc
   */
  public function setMediaStatusPublishedValue(string $publishedValue): ConverterInterface {
    $this->set('media_status_published_value', $publishedValue);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setStatusManagement(string $statusManagement): ConverterInterface {
    if (!in_array($statusManagement, [
      ConverterInterface::STATUS_MANAGEMENT_DELETE,
      ConverterInterface::STATUS_MANAGEMENT_UNPUBLISH
    ], TRUE)) {
      throw new \InvalidArgumentException('The given status management method is not valid.');
    }

    $this->set('status_management', $statusManagement);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getGid(): array {
    return $this->get('gid') ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function setGid(array $gid): ConverterInterface {
    $this->set('gid', $gid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMapsType(): ?string {
    return $this->get('maps_type');
  }

  /**
   * {@inheritdoc}
   */
  public function setMapsType(string $type): ConverterInterface {
    $this->set('maps_type', $type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getSourceHandler(): SourceHandlerInterface {
    $sourceHandlerPluginManager = \Drupal::service('plugin.manager.maps_sync_source_handler');
    return $sourceHandlerPluginManager->createInstance($this->getMapsType());
  }

  /**
   * {@inheritdoc}
   */
  public function getHandler(): MapsSyncHandlerInterface {
    $plugins = \Drupal::service('plugin.manager.maps_sync_handler');
    return $plugins->createInstance($this->getHandlerId());
  }

  /**
   * {@inheritdoc}
   */
  public function getHandlerId(): ?string {
    return $this->get('handler_id');
  }

  /**
   * {@inheritdoc}
   */
  public function setHandlerId(string $handler_id): ConverterInterface {
    $this->set('handler_id', $handler_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getParent(): ?string {
    return $this->get('parent') ?? 'default';
  }

  /**
   * {@inheritdoc}
   */
  public function setParent(string $parent): ConverterInterface {
    $this->set('parent', $parent);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): ?int {
    return $this->get('weight');
  }

  /**
   * {@inheritdoc}
   */
  public function setWeight(int $weight): ConverterInterface {
    $this->set('weight', $weight);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function importEntity(EntityInterface $entity) {
    $object = $this->getSourceHandler()->findObjectFromEntity($this,$entity);
    return $this->importObject($object);
  }

  /**
   * {@inheritdoc}
   */
  public function importObject(BaseInterface $object) {
    $logger = \Drupal::logger('a12s_maps_sync');
    $logger->debug(
      "Importing {$this->getMapsType()} {$object->getId()} with converter {$this->id()}",
      [
        'converter' => $this->id(),
        'object_id' => $object->getId(),
      ]
    );

    // Ensure that the object matches the converter's filters.
    if (!$this->objectMatchFilters($object)) {
      return [];
    }

    $results = [];
    $event_dispatcher = \Drupal::service('event_dispatcher');

    // Check if the entity is translatable.
    $config = ContentLanguageSettings::loadByEntityTypeBundle($this->getConverterEntityType(), $this->getConverterBundle());

    $default_language = \Drupal::languageManager()->getDefaultLanguage();
    if ($config->getThirdPartySetting('content_translation', 'enabled')) {
      $languages = \Drupal::languageManager()->getLanguages();

      // Set the default language first.
      unset($languages[$default_language->getId()]);

      $languages = array_values($languages);
      array_unshift($languages, $default_language);
    }
    else {
      // @todo Use the default language? Or "und"?
      $languages = [\Drupal::languageManager()->getDefaultLanguage()];
    }

    foreach ($languages as $language) {
      try {
        $logger->debug(
          "Converting {$this->getMapsType()} {$object->getId()} with converter {$this->id()}",
          [
            'converter' => $this->id(),
            'object_id' => $object->getId(),
            'language' => $language,
          ]
        );

        $entity = $this->getHandler()->convertItem($object, $this, $language);

        if ($entity) {
          $result['gid'] = $entity->get(BaseInterface::GID_FIELD)->value;
          $result['maps_object'] = $object;

          if ($entity) {
            // Add context to the entity for contributed modules and allow for
            // modifications of the $result array.
            $entity->context['a12s_maps_sync'] = &$result;

            $this->getHandler()->preSave($entity, $object, $this, $language);
            $result['success'] = $entity->save();
            $this->getHandler()->postSave($entity, $object, $this, $language);

            $result['entity_id'] = $entity->id();

            $event = new ConverterItemImportEvent($this, $object, $entity, $language);
            $event_dispatcher->dispatch($event, ConverterItemImportEvent::FINISHED);
          }
        }
      } catch (\Exception $e) {
        \Drupal::logger('a12s_maps_sync')->error(
          "Error during import",
          [
            'converter' => $this->id(),
            'object' => $object->getId(),
            'exception' => $e->getMessage(),
          ]
        );
        watchdog_exception('a12s_maps_sync', $e);

        return FALSE;
      } finally {
        if (isset($result)) {
          $results[] = $result['gid'];
        }
      }
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function import(int $limit): array {
    \Drupal::logger('a12s_maps_sync')->debug("Importing converter {$this->id()} (limit: $limit)");

    $results = [];
    $filters = $this->getFiltersArray();

    /** @var \Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginManager $sourceHandlerManager */
    $sourceHandlerManager = \Drupal::service('plugin.manager.maps_sync_source_handler');

    /** @var \Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginBase $sourceHandler */
    $sourceHandler = $sourceHandlerManager->createInstance($this->getMapsType());

    $data = $sourceHandler->getData($this, $filters, $limit, $this->getLastImportedTime());

    if (!empty($data)) {
      foreach ($data as $object) {
        if ($_results = $this->importObject($object)) {
          $results = array_merge($results, $_results);
        }
      }

      $this->setLastImportedTime($object->getImported());
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function rollback() {
    $entity_type = \Drupal::entityTypeManager()->getDefinition($this->getConverterEntityType());

    $query = \Drupal::entityQuery($this->getConverterEntityType());

    $bundle = $this->getConverterBundle();
    if ($bundle !== $entity_type->id()) {
      $query->condition($entity_type->getKey('bundle'), $bundle);
    }

    $deleted = [];

    $query->accessCheck(FALSE);
    $results = $query->execute();

    foreach ($results as $id) {
      $entity = \Drupal::entityTypeManager()->getStorage($entity_type->id())->load($id);

      if ($entity !== NULL && $entity->hasField(BaseInterface::GID_FIELD) && $entity->get(BaseInterface::GID_FIELD)->value) {
        try {
          $entity->delete();
          $deleted[] = $id;
        }
        catch (EntityStorageException $e) {
          watchdog_exception('a12s_maps_sync', $e);
        }
      }
    }

    return $deleted;
  }

  /**
   * {@inheritdoc}
   */
  protected function urlRouteParameters($rel): array {
    $uri_route_parameters = parent::urlRouteParameters($rel);
    $uri_route_parameters['maps_sync_profile'] = $this->getProfile()->id();
    return $uri_route_parameters;
  }

  /**
   * @return string
   */
  public function getLastImportedStateKey(): string {
    return 'a12s_maps_sync_converter_last_importer_' . $this->id();
  }

  /**
   * {@inheritdoc}
   */
  public function getLastImportedTime(): int {
    return \Drupal::state()->get($this->getLastImportedStateKey(), 0);
  }

  /**
   * {@inheritdoc}
   */
  public function setLastImportedTime(int $time): void {
    \Drupal::state()->set($this->getLastImportedStateKey(), $time);
  }

  /**
   * {@inheritdoc}
   */
  public function resetLastImportedTime(): void {
    \Drupal::state()->delete($this->getLastImportedStateKey());
  }

  /**
   * {@inheritdoc}
   */
  public function getLockName(): string {
    return 'a12s_maps_sync:lock:profile:' . $this->getProfile()->id();
  }

  /**
   * {@inheritdoc}
   */
  public function addMappingItem(Mapping $mapping): Converter {
    $current = $this->get('mapping');
    $current[] = $mapping->toArray();
    $this->set('mapping', $current);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConverterDependencies(array &$dependencies = []): array {
    // Parse the mapping.
    foreach ($this->getMapping() as $mapping) {
      if ($converters = $mapping->getConverters()) {
        foreach ($converters as $converter) {
          $dependencies = [$converter->id() => $converter] + $dependencies;
          $converter->getConverterDependencies($dependencies);
        }
      }
    }

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    if ($profile = $this->getProfile()) {
      $weights = [];

      // We have to ensure that the weight is unique.
      foreach ($profile->getConverters() as $converter) {
        if ($converter->id() === $this->id()) {
          continue;
        }

        $weights[] = $converter->getWeight();
      }

      while (in_array($this->getWeight(), $weights)) {
        $this->setWeight($this->getWeight() + 1);
      }
    }

    parent::preSave($storage);
  }

  /**
   * @param \Drupal\a12s_maps_sync\Maps\BaseInterface $object
   *
   * @return array
   */
  public function getObjectDependencies(BaseInterface $object): array {
    /** @var \Drupal\a12s_maps_sync\Plugin\MappingHandlerPluginManager $mappingHandlerPluginManager */
    $mappingHandlerPluginManager = \Drupal::service('plugin.manager.maps_sync_mapping_handler');

    $return = [];

    foreach ($this->getMapping() as $mapping) {
      if ($converters = $mapping->getConverters()) {
        foreach ($converters as $converter) {
          $values = [];

          for ($i = 0; $i < $object->getCountValues($mapping->getSource()); $i++) {
            foreach ($converter->getGid() as $elem) {
              if (in_array($elem, $this->getSourceHandler()->getAllowedGidEntityKeys())) {
                $mappingHandler = $mappingHandlerPluginManager->createInstance($mapping->getHandlerId());

                $values[] = $mappingHandler->getObjectValue($converter, $object, $mapping, $i);
              }
            }
          }

          $return[] = [
            'values' => $values,
            'converter' => $converter,
            'with_dependencies' => $mapping->getHandlerId() === 'link',
          ];
        }
      }
    }

    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function objectMatchFilters(BaseInterface $object): bool {
    $filters = $this->getFilters();

    // Avoid importing all the MaPS database...
    if (empty($filters)) {
      return FALSE;
    }

    foreach ($filters as $filter) {
      if ($filter->getType() === 'class') {
        if (!in_array($filter->getValue(), $object->getClasses())) {
          return FALSE;
        }
      }
      else {
        // Some preprocess.
        // @todo move this in some plugins.
        $type = match($filter->getType()) {
          'object_type', 'media_type' => 'type',
          'object_nature' => 'nature',
          'library' => 'id_attribute',
          default => $filter->getType(),
        };

        if (str_starts_with($type, 'attribute_')) {
          $type = str_replace('attribute_', '', $type);
        }

        $attributeValue = $object->get($type);
        if ($filter->getFilteringType() === 'exist' && !$attributeValue) {
          return FALSE;
        }

        if ($filter->getFilteringType() === 'not_exist' && $attributeValue) {
          return FALSE;
        }

        if ($filter->getFilteringType() === 'empty' && $attributeValue) {
          return FALSE;
        }

        // Get the value.
        $values = [$object->get($type)];

        // If we have additional data, we'll retrieve the code (as for libraries for example).
        $attribute = $object->getAttribute($type);
        if ($attribute && $data = $attribute->getData()) {
          $datum = reset($data);
          foreach (['code', 'objectCode'] as $key) {
            if (!empty($datum) && !empty($datum[$key])) {
              $values[] = $datum[$key];
            }
          }
        }

        $filterValue = $filter->getValue();

        // Another specific case.
        // For some reason, MaPS uses only codes in the configuration, but not for media types...
        if ($filter->getType() === 'media_type') {
          $filterValue = match($filterValue) {
            'img' => 1,
            'doc' => 2,
            default => NULL,
          };
        }

        if ($filter->getFilteringType() === 'value' && !in_array($filterValue, $values)) {
          return FALSE;
        }
      }
    }

    return TRUE;
  }

}
