<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\Entity\Profile;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class ConfigExplorerForm extends FormBase {

  /**
   * @param \Drupal\a12s_maps_sync\MapsApi $mapsApi
   */
  public function __construct(
    protected MapsApi $mapsApi,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('a12s_maps_sync.maps_api')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'a12s_maps_sync.config_explorer_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['filters'] = ['#type' => 'fieldset'];

    //'id', 'code', 'type', 'id_language']
    $options = [];
    foreach (Profile::loadMultiple() as $profile) {
      $options[$profile->getPythonProfileId()] = $profile->label();
    }

    $form['filters']['profile'] = [
      '#type' => 'select',
      '#title' => $this->t('Profile'),
      '#options' => $options,
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
    ];

    $form['filters']['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Type'),
      '#options' => [
        'attribute' => $this->t('Attribute'),
        'attribute_set' => $this->t('Attribute set'),
        'attribute_set_has_attribute' => $this->t('Attribute set has attribute'),
        'class' => $this->t('Class'),
        'library' => $this->t('Library'),
        'media_type' => $this->t('Media type'),
        'object_class_has_attribute_set' => $this->t('Object class has attribute set'),
        'object_nature' => $this->t('Object nature'),
        'object_type' => $this->t('Object type'),
      ],
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
    ];

    $form['filters']['id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ID'),
    ];

    $form['filters']['code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Code'),
    ];

    $languages = \Drupal::config('a12s_maps_sync.languages_mapping')->get();
    $options = [];
    foreach ($languages as $langcode => $id) {
      $options[$id] = $langcode;
    }

    $form['filters']['id_language'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#options' => $options,
      '#default_value' => 1,
    ];

    $form['filters']['actions'] = ['#type' => 'actions'];
    $form['filters']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Search'),
    ];

    $values = $form_state->cleanValues()->getValues();
    if (!empty($values['results'])) {
      $form['results'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Type'),
          $this->t('ID'),
          $this->t('Code'),
          $this->t('Value'),
          $this->t('Extra data'),
        ],
        '#empty' => $this->t('No result found'),
      ];

      foreach ($values['results'] as $result) {
        $form['results'][] = [
          'type' => [
            '#plain_text' => $result['type'],
          ],
          'id' => [
            '#plain_text' => $result['id'],
          ],
          'code' => [
            '#plain_text' => $result['code'],
          ],
          'value' => [
            '#plain_text' => $result['value'],
          ],
          'data' => [
            '#plain_text' => Json::encode($result['data']),
          ],
        ];
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $filters = $form_state->cleanValues()->getValues();

    $results = [];
    if (!empty($filters['profile'])) {
      $profile = Profile::load($filters['profile']);
      unset($filters['profile']);

      $results = $this->mapsApi->getConfiguration($profile, $filters);
    }

    $form_state->setRebuild(TRUE);
    $form_state->setValue('results', $results);
  }

}
