<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\AutoConfigManager;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\a12s_maps_sync\Plugin\SourceHandlerPluginManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ConverterAutoConfigForm.
 */
class ConverterAutoConfigForm extends EntityForm {

  /**
   * @param \Drupal\a12s_maps_sync\MapsApi $mapsApi
   */
  public function __construct(
    protected MapsApi $mapsApi,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('a12s_maps_sync.maps_api'),
    );
  }

  /**
   * Title callback.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   */
  public function title(): TranslatableMarkup {
    return $this->t("Automatic configuration for converter %converter", ['%converter' => \Drupal::request()->get('maps_sync_converter')->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
    $converter = $this->entity;

    $mapsAttributeSets = $this->mapsApi->getConfiguration($converter->getProfile(), ['type' => 'attribute_set']);
    $options = [];
    foreach ($mapsAttributeSets as $mapsAttributeSet) {
      $options[$mapsAttributeSet['code']] = $mapsAttributeSet['value'];
    }

    $form['attribute_sets'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Attribute sets'),
      '#prefix' => '<div id="attribute-sets-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    $autoConfig = $converter->getAutoConfig();
    $attributeSets = $autoConfig['attribute_sets'] ?? [];
    $attributesDenyList = $autoConfig['attributes_deny_list'] ?? [];
    $librariesManagement = $autoConfig['libraries_management'] ?? '';

    $countAttributeSets = $form_state->get('count_attribute_sets');
    if ($countAttributeSets === NULL) {
      $countAttributeSets = max(count($attributeSets), 1);

      $form_state->set('count_attribute_sets', $countAttributeSets);
    }

    for ($i = 0; $i < $countAttributeSets; $i++) {
      $attributeSet = !empty($attributeSets[$i]) ? $attributeSets[$i] : NULL;

      $form['attribute_sets'][$i] = [
        '#type' => 'select',
        '#options' => $options,
        '#empty_option' => $this->t('- Select -'),
        '#default_value' => $attributeSet,
      ];
    }

    $form['attribute_sets']['actions'] = [
      '#type' => 'actions',
    ];

    $form['attribute_sets']['actions']['add_attribute_set'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add attribute set'),
      '#submit' => ['::addAttributeSet'],
      '#ajax' => [
        'callback' => '::addAttributeSetCallback',
        'wrapper' => 'attribute-sets-wrapper',
      ],
    ];

    $form['attributes_deny_list'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Attributes deny list'),
      '#description' => $this->t('A list of attributes to ignore. One attribute code by line.'),
      '#default_value' => !empty($attributesDenyList) ? implode("\r\n", $attributesDenyList) : '',
    ];

    $form['libraries_management'] = [
      '#type' => 'select',
      '#title' => $this->t('Libraries management'),
      '#description' => $this->t('Select the way to manage libraries (entity references fields or list fields)'),
      '#default_value' => $librariesManagement,
      '#options' => [
        AutoConfigManager::LIBRARIES_MANAGEMENT_ENTITY_REFERENCE => $this->t('Entity reference fields'),
        AutoConfigManager::LIBRARIES_MANAGEMENT_LIST => $this->t('List fields'),
      ],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): void {
    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
    $converter = $this->entity;
    $status = $converter->save();

    \Drupal::messenger()->addMessage($this->t('Saved auto config for the the %label converter.', [
      '%label' => $converter->label(),
    ]));

    $form_state->setRedirectUrl($converter->toUrl('collection'));
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    unset($values['attribute_sets']['actions']);

    $autoConfig = $entity->getAutoConfig();
    $autoConfig['attribute_sets'] = array_filter($values['attribute_sets']);
    $autoConfig['attributes_deny_list'] = explode("\r\n", $values['attributes_deny_list']);
    $autoConfig['libraries_management'] = $values['libraries_management'];

    $entity->setAutoConfig($autoConfig);
  }

  /**
   * Submit handler for the "add-attribute-set" button.
   *
   * Increments the max counter and causes a rebuild.
   */
  public function addAttributeSet(array &$form, FormStateInterface $form_state): void {
    $name_field = $form_state->get('count_attribute_sets');
    $add_button = $name_field + 1;
    $form_state->set('count_attribute_sets', $add_button);
    $form_state->setRebuild();
  }

  /**
   * Callback for both ajax-enabled buttons.
   *
   * Selects and returns the fieldset with the names in it.
   */
  public function addAttributeSetCallback(array &$form, FormStateInterface $form_state) {
    return $form['attribute_sets'];
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    return [
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#submit' => ['::submitForm', '::save'],
      ]
    ];
  }

}
