<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\Converter\Mapping;
use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Exception\MapsApiException;
use Drupal\a12s_maps_sync\MapsApi;
use Drupal\a12s_maps_sync\Plugin\MappingHandlerPluginManager;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Serialization\Yaml;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use GuzzleHttp\Exception\GuzzleException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ConverterMappingForm.
 */
class ConverterMappingForm extends EntityForm {

  /**
   * @var array
   */
  protected array $mapsAttributes = [];

  /**
   * @param MapsApi $mapsApi
   * @param EntityFieldManagerInterface $entityFieldManager
   * @param MappingHandlerPluginManager $mappingHandlerPluginManager
   */
  public function __construct(
    protected MapsApi $mapsApi,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected MappingHandlerPluginManager $mappingHandlerPluginManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('a12s_maps_sync.maps_api'),
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.maps_sync_mapping_handler'),
    );
  }

  /**
   * Title callback.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   */
  public function title(): TranslatableMarkup {
    return $this->t("Edit mapping for converter %converter", ['%converter' => \Drupal::request()->get('maps_sync_converter')->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\a12s_maps_sync\Entity\ConverterInterface $converter */
    $converter = $this->entity;

    // Mapping management.
    $form['mapping'] = [
      '#type' => 'table',
      '#title' => $this->t('Mapping'),
      '#prefix' => '<div id="mapping-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
      '#header' => [
        $this->t('Settings'),
        $this->t('Options'),
      ],
    ];

    $mapping = $converter->getMapping();

    $countMapping = $form_state->get('count_mapping');
    if ($countMapping === NULL) {
      $countMapping = max(count($mapping), 1);

      $form_state->set('count_mapping', $countMapping);
    }

    for ($i = 0; $i < $countMapping; $i++) {
      $mappingItem = !empty($mapping[$i]) ? $mapping[$i] : NULL;

      $form['mapping'][$i] = [
        '#attributes' => [
          'id' => "mapping-$i-wrapper",
        ],
      ];

      $status = !is_null($mappingItem) ? $mappingItem->getStatus() : Mapping::MAPPING_STATUS_MANUAL;
      $statusLabel = match ($status) {
        Mapping::MAPPING_STATUS_MANUAL => $this->t('Manually created'),
        Mapping::MAPPING_STATUS_AUTO => $this->t('Automatically created'),
        Mapping::MAPPING_STATUS_OVERRIDE => $this->t('Override'),
      };

      $form['mapping'][$i]['settings']['status'] = [
        '#type' => 'hidden',
        '#default_value' => $status,
      ];

      $form['mapping'][$i]['settings']['status_label'] = [
        '#markup' => $statusLabel,
      ];

      $form['mapping'][$i]['settings']['source'] = [
        '#type' => 'select',
        '#title' => $this->t('Source'),
        '#empty_option' => $this->t('- Select -'),
        '#options' => $this->getMappingSourceOptions(),
        '#default_value' => !is_null($mappingItem) ? $mappingItem->getSource() : NULL,
      ];

      $form['mapping'][$i]['settings']['target'] = [
        '#type' => 'select',
        '#title' => $this->t('Target'),
        '#empty_option' => $this->t('- Select -'),
        '#options' => $this->getMappingTargetOptions(),
        '#default_value' => !is_null($mappingItem) ? $mappingItem->getTarget() : NULL,
      ];


      $form['mapping'][$i]['settings']['required'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Required'),
        '#default_value' => !is_null($mappingItem) ? $mappingItem->isRequired() : NULL,
      ];

      $form['mapping'][$i]['settings']['requiredBehavior'] = [
        '#type' => 'select',
        '#title' => $this->t('Required behavior'),
        '#default_value' => !is_null($mappingItem) ? $mappingItem->getRequiredBehavior() : NULL,
        '#options' => [
          Mapping::MAPPING_REQUIRED_ERROR => 'Throw error',
          Mapping::MAPPING_REQUIRED_SKIP => 'Skip object'
        ],
        '#states' => [
          'visible' => [
            ':input[name="mapping[' . $i . '][settings][required]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['mapping'][$i]['settings']['append'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Append'),
        '#default_value' => !is_null($mappingItem) ? $mappingItem->isAppend() : FALSE,
      ];

      $mappingHandlerId = !is_null($mappingItem) && !is_null($mappingItem->getHandlerId()) ? $mappingItem->getHandlerId() : ($form_state->getUserInput()['mapping'][$i]['handler'] ?? 'default');

      $config = !is_null($mappingItem) ? $mappingItem->getOptions() : [];
      $config['source_converter'] = $converter;

      $mappingHandler = $this->mappingHandlerPluginManager->createInstance(
        $mappingHandlerId,
        $config,
      );

      $subForm = [];
      $subFormState = SubformState::createForSubform($subForm, $form, $form_state);
      $subForm = $mappingHandler->buildConfigurationForm($subForm, $subFormState);

      $form['mapping'][$i]['configuration'] = $subForm;

      $form['mapping'][$i]['configuration']['handler'] = [
        '#type' => 'select',
        '#title' => $this->t('Handler'),
        '#empty_option' => $this->t('- Select -'),
        '#options' => $this->getMappingHandlerOptions(),
        '#default_value' => $mappingHandlerId,
        '#ajax' => [
          'wrapper' => 'mapping-wrapper',
          'callback' => '::refreshMappingItemCallback',
        ],
        '#weight' => -50,
      ];
    }

    $form['mapping'][$i + 1]['actions'] = [
      '#type' => 'actions',
      '#wrapper_attributes' => ['colspan' => 6],
    ];

    $form['mapping'][$i + 1]['actions']['add_filter'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add mapping item'),
      '#submit' => ['::addMappingItem'],
      '#ajax' => [
        'callback' => '::addMappingItemCallback',
        'wrapper' => 'mapping-wrapper',
      ],
    ];

    return $form;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    $converterMapping = $entity->getMapping();

    // Some post process.
    unset($values['mapping']['actions']);
    $mapping = [];

    foreach ($values['mapping'] as $i => $mappingData) {
      // Format data from the form.
      if (!empty($mappingData['settings'])) {
        foreach ($mappingData['settings'] as $k => $v) {
          $mappingData[$k] = $v;
        }
        unset($mappingData['settings']);
      }

      if (!empty($mappingData['source']) && !empty($mappingData['target'])) {
        if (!empty($mappingData['configuration'])) {
          $mappingData['handler'] = $mappingData['configuration']['handler'];
          unset($mappingData['configuration']['handler']);

          // Cast some values.
          $mappingData['required'] = (bool) $mappingData['required'];
          $mappingData['requiredBehavior'] = (int) $mappingData['requiredBehavior'];
          $mappingData['append'] = (bool) $mappingData['append'];

          $mappingData['options'] = [];
          foreach ($mappingData['configuration'] as $k => $v) {
            if (is_array($v)) {
              $v = array_filter($v);
            }

            $mappingData['options'][$k] = $v;
          }
          unset($mappingData['configuration']);
        }

        // If the mapping item has been created automatically, we have to check
        // if it has been override.
        if ($mappingData['status'] == Mapping::MAPPING_STATUS_AUTO) {
          // Since there is no identifier on mapping items, and the mapping items
          // are not movable, we will compare submitted values against existing
          // mapping.
          if (!empty($converterMapping[$i])) {
            $currentMapping = $converterMapping[$i]->toArray();
            foreach ($mappingData as $k => $v) {
              if (is_array($v)) {
                $v = array_filter($v);
                $currentMapping[$k] = array_filter($currentMapping[$k]);
              }

              // Avoid empty form values / no configuration.
              if (empty($v) && empty($currentMapping[$k])) {
                continue;
              }

              // Different values, update the status.
              if ($currentMapping[$k] != $v) {
                $mappingData['status'] = Mapping::MAPPING_STATUS_OVERRIDE;
              }
            }
          }
        }

        $mapping[] = $mappingData;
      }
    }

    $entity->set('mapping', $mapping);
  }

  /**
   * {@inheritdoc}
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): void {
    $maps_sync_converter = $this->entity;
    $maps_sync_converter->save();

    \Drupal::messenger()->addMessage($this->t('Saved mapping for the the %label converter.', [
      '%label' => $maps_sync_converter->label(),
    ]));

    $form_state->setRedirectUrl($maps_sync_converter->toUrl('collection'));
  }

  /**
   * @return array
   */
  public function getMappingSourceOptions(): array {
    /** @var ConverterInterface $converter */
    $converter = $this->entity;

    $sourceHandler = $converter->getSourceHandler();
    $options = $sourceHandler->getMappingSources();

    if ($sourceHandler->hasAttributes()) {
      // Add attributes.
      $options['Attributes'] = [];
      try {
        if (empty($this->mapsAttributes)) {
          $this->mapsAttributes = $this->mapsApi->getConfiguration($converter->getProfile(), ['type' => 'attribute']);
        }

        foreach ($this->mapsAttributes as $attribute) {
          $options['Attributes'][$attribute['code']] = $attribute['value'] . " ({$attribute['code']})";
        }

        asort($options['Attributes']);
      } catch (MapsApiException|GuzzleException $e) {
        watchdog_exception('a12s_maps_sync', $e);
      }
    }

    return $options;
  }

  /**
   * @return array
   */
  public function getMappingTargetOptions(): array {
    /** @var ConverterInterface $converter */
    $converter = $this->entity;

    $fields = $this->entityFieldManager->getFieldDefinitions(
      $converter->getConverterEntityType(),
      $converter->getConverterBundle()
    );

    $options = [];
    foreach ($fields as $id => $field) {
      $options[$id] = $field->getLabel() . " ($id)";
    }

    asort($options);

    return $options;
  }

  /**
   * Submit handler for the "add-mapping-item" button.
   *
   * Increments the max counter and causes a rebuild.
   */
  public function addMappingItem(array &$form, FormStateInterface $form_state): void {
    $name_field = $form_state->get('count_mapping');
    $add_button = $name_field + 1;
    $form_state->set('count_mapping', $add_button);
    $form_state->setRebuild();
  }

  /**
   * Callback for both ajax-enabled buttons.
   *
   * Selects and returns the fieldset with the names in it.
   */
  public function addMappingItemCallback(array &$form, FormStateInterface $form_state) {
    return $form['mapping'];
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @return void
   */
  public function refreshMappingItemCallback(array &$form, FormStateInterface $form_state) {
    return $form['mapping'];
  }

  /**
   * @return array
   */
  protected function getMappingHandlerOptions(): array {
    $options = [];
    foreach ($this->mappingHandlerPluginManager->getDefinitions() as $definition) {
      $options[$definition['id']] = $definition['label'];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    return [
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#submit' => ['::submitForm', '::save'],
      ]
    ];
  }

}
