<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\BatchService;
use Drupal\a12s_maps_sync\Entity\Converter;
use Drupal\a12s_maps_sync\Entity\Profile;
use Drupal\a12s_maps_sync\Exception\MapsException;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

class ReimportEntityForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'a12s_maps_sync.reimport_entity_form';
  }

  /**
   * @return \Drupal\Core\Entity\EntityInterface|null
   */
  protected function getEntityFromRoute(): ?EntityInterface {
    $parameter_name = \Drupal::routeMatch()->getRouteObject()->getOption('_a12s_entity_type_id');
    return \Drupal::routeMatch()->getParameter($parameter_name);
  }

  /**
   * Title callback.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   */
  public function title(): TranslatableMarkup {
    $entity = $this->getEntityFromRoute();

    return $this->t("Reimport %entity_type %entity", [
      '%entity_type' => $entity->getEntityType()->getLabel(),
      '%entity' => $entity->label(),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $entity = $this->getEntityFromRoute();

    if (!$entity->hasField(BaseInterface::GID_FIELD)) {
      $error = $this->t('This entity can not be imported by MaPS.');
    }

    if ($entity->get(BaseInterface::GID_FIELD)->isEmpty()) {
      $error = $this->t('This entity has not been imported by MaPS.');
    }

    if (isset($error)) {
      $form['error'] = [
        '#markup' => $error,
        '#attributes' => [
          'class' => ['error'],
        ],
      ];

      return $form;
    }

    $form_state->set('entity', $entity);

    $form['intro'] = [
      '#markup' => $this->t('This form allows you to re-import the given entity.'),
    ];

    $form['#prefix'] = '<div id="import-form-wrapper">';
    $form['#suffix'] = '</div>';

    $profileOptions = [];
    foreach (Profile::loadMultiple() as $profile) {
      $profileOptions[$profile->getPythonProfileId()] = $profile->label();
    }

    $form['profile'] = [
      '#type' => 'select',
      '#title' => $this->t('Profile'),
      '#options' => $profileOptions,
      '#description' => $this->t('Specify the profile to use to import the entity. If not specified, it will be detected automatically.'),
      '#ajax' => [
        'callback' => '::updateConvertersValues',
        'event' => 'change',
        'wrapper' => 'import-form-wrapper',
      ],
      '#empty_option' => $this->t('- Select -'),
    ];

    $userInput = $form_state->getUserInput();
    if (!empty($userInput['profile'])) {
      $converterOptions = [];

      $profile = Profile::load($userInput['profile']);
      foreach ($profile->getConverters() as $converter) {
        $converterOptions[$converter->id()] = $converter->label();
      }

      $form['converter'] = [
        '#type' => 'select',
        '#title' => $this->t('Converter'),
        '#options' => $converterOptions,
        '#description' => $this->t('Specify the profile to use to import the entity. If not specified, it will be detected automatically.'),
      ];
    }

    $form['with_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('With dependencies'),
      '#default_value' => TRUE,
      '#description' => $this->t('Import the dependencies. It is not advised to uncheck this, as it can lead to errors or unexpected results'),
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Re-import'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var EntityInterface $entity */
    $entity = $form_state->get('entity');

    $batch = BatchService::getEntityImportBatchDefinition(
      $entity,
      (bool) $form_state->getValue('with_dependencies'),
      $form_state->getValue('profile') ? Profile::load($form_state->getValue('profile')) : NULL,
      $form_state->getValue('converter') ? Converter::load($form_state->getValue('converter')) : NULL,
    );

    if ($batch) {
      $form_state->setRedirect("entity.{$entity->getEntityTypeId()}.canonical", [$entity->getEntityTypeId() => $entity->id()]);

      batch_set($batch);
    }
    else {
      throw new MapsException("Unable to import the given entity.");
    }
  }

  /**
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $formState
   *
   * @return array
   */
  public function updateConvertersValues(array &$form, FormStateInterface $formState): array {
    return $form;
  }

}
