<?php

namespace Drupal\a12s_maps_sync\Plugin\MapsSyncHandler;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Event\ConverterItemImportEvent;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Class MediaHandler
 *
 * Handler used for files when we want to create the related media on the fly.
 *
 * @MapsSyncHandler(
 *   id = "media",
 *   label = @Translation("Media")
 * )
 */
class MediaHandler extends MapsSyncHandlerBase {

  /**
   * Get the media types mapping.
   *
   * @return array
   */
  public function getMediaTypesMapping(): array {
    return [
      1 => [
        'type' => 'image',
        'field' => 'field_image',
      ],
      2 => [
        'type' => 'document',
        'field' => 'field_document_file',
      ],
    ];
  }

  /**
   * Get the media type mapping for a given object.
   *
   * @param \Drupal\a12s_maps_sync\Maps\BaseInterface $object
   *
   * @return array
   */
  public function getMediaTypeMapping(BaseInterface $object): array {
    return $this->getMediaTypesMapping()[$object->get('type')];
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL) {
    // Create the media.
    // To find the media type, we are using the media type from MaPS.
    $media_type_mapping = $this->getMediaTypeMapping($object);

    // Check if the entity exists.
    $sourceHandler = $converter->getSourceHandler();
    $media = $sourceHandler->getEntity('media', $media_type_mapping['type'], $object, $converter->getGid(), $language);

    $media->{$media_type_mapping['field']}->target_id = $entity->id();
    $media->set(BaseInterface::GID_FIELD, $entity->get(BaseInterface::GID_FIELD)->value);

    // Specific case for images, we have to set the "alt" property of the image field.
    if ($media_type_mapping['type'] === 'image') {
      $_language = $language !== NULL && $entity->isTranslatable()
        ? \Drupal::config('a12s_maps_sync.languages_mapping')
          ->get($language->getId())
        : $converter->getProfile()->getDefaultMapslanguage();

      $media->{$media_type_mapping['field']}->alt = $object->get('MEDIALIBELLE', $_language);
    }

    $media->save();

    // We need to retrigger the dispatch of the item import finished event here.
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event = new ConverterItemImportEvent($converter, $object, $media, $language);
    $event_dispatcher->dispatch($event, ConverterItemImportEvent::FINISHED);
  }

  /**
   * @inheritDoc
   */
  public function postDelete(EntityInterface $entity, BaseInterface $object, ConverterInterface $converter, LanguageInterface $language = NULL): void {
    $media_type_mapping = $this->getMediaTypeMapping($object);

    // Check if the entity exists.
    $sourceHandler = $converter->getSourceHandler();
    $media = $sourceHandler->getEntity('media', $media_type_mapping['type'], $object, $converter->getGid(), $language);
    $media->delete();
  }

}
