<?php

namespace Drupal\a12s_maps_sync\Plugin;

use Drupal\a12s_maps_sync\Entity\ConverterInterface;
use Drupal\a12s_maps_sync\Entity\ProfileInterface;
use Drupal\a12s_maps_sync\Exception\MapsApiException;
use Drupal\a12s_maps_sync\Maps\Attribute;
use Drupal\a12s_maps_sync\Maps\BaseInterface;
use Drupal\a12s_maps_sync\Maps\Exception\MapsEntityDefinitionException;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryException;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Base class for a12s_maps_sync_source_handler plugins.
 */
abstract class SourceHandlerPluginBase extends PluginBase implements SourceHandlerInterface {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function label() {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFilterValues(ConverterInterface $converter, string $type): array {
    $profile = $converter->getProfile();

    try {
      /** @var \Drupal\a12s_maps_sync\MapsApi $mapsApi */
      $mapsApi = \Drupal::service('a12s_maps_sync.maps_api');
      $config = $mapsApi->getConfiguration($profile, ['type' => $type]);
    } catch (MapsApiException|GuzzleException $e) {
      watchdog_exception('a12s_maps_sync', $e);
      return [];
    }

    $options = [];
    foreach ($config as $item) {
      $options[$item['code']] = $item['value'] . " ({$item['code']})";
    }

    return $options;
  }


  /**
   * {@inheritdoc}
   */
  public function getEntity(string $entity_type, string $bundle = NULL, BaseInterface $object, array $gid_definition, LanguageInterface $language = NULL) {
    try {
      if (empty($gid_definition)) {
        throw new MapsEntityDefinitionException($entity_type, 'gid');
      }

      $langCode = $language?->getId() ?? LanguageInterface::LANGCODE_NOT_SPECIFIED;

      // @todo how to handle object translatable attributes?
      if ($gid = $this->getGid($gid_definition, $object)) {
        $storage = \Drupal::entityTypeManager()->getStorage($entity_type);
        $entities = $storage->loadByProperties([BaseInterface::GID_FIELD => $gid]);
        // @todo load translation?

        // @todo how to handle multiple results? This should never happen!
        if ($entities) {
          $entity = reset($entities);
          if (in_array($langCode, [
            \Drupal::languageManager()->getDefaultLanguage()->getId(),
            LanguageInterface::LANGCODE_NOT_SPECIFIED,
          ], TRUE)) {
            return $entity;
          }

          return $entity->hasTranslation($langCode) ? $entity->getTranslation($langCode) : $entity->addTranslation($langCode);
        }

        $time = \Drupal::time()->getRequestTime();
        $params = [
          BaseInterface::GID_FIELD => $gid,
          // @todo Use $definition['translatable'] and force "und" langcode?
          'langcode' => $langCode,
          'created' => $time,
          'changed' => $time,
        ];

        if ($bundle !== NULL) {
          /** @var EntityTypeManagerInterface $entity_type_manager */
          $entity_type_manager = \Drupal::service('entity_type.manager');
          $type = $entity_type_manager->getDefinition($entity_type);

          $params[$type->getKey('bundle')] = $bundle;
        }

        return $storage->create($params);
      }
    } catch (PluginNotFoundException $e) {
      \Drupal::logger('a12s_maps_sync')->error('The entity type %entity_type is not defined.', ['%entity_type' => $entity_type]);
    } catch (MapsEntityDefinitionException $e) {
      watchdog_exception('a12s_maps_sync', $e);
    } catch (InvalidPluginDefinitionException $e) {
      // Should never happen, as if the entity definition is found, the entity
      // storage plugin exists.
      watchdog_exception('a12s_maps_sync', $e);
    } catch (QueryException $e) {
      // Seems that the "a12s_maps_sync_gid" field in missing in the entity table.
      watchdog_exception('a12s_maps_sync', $e);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getGid(array $definition, BaseInterface $object): false|string {
    $gid = [];

    foreach ($definition as $item) {
      $exploded = explode(':', $item);

      // If the element begins with "const:", it is a constant, and we don't
      // have to get any value...
      if (count($exploded) === 2 && $exploded[0] === 'const') {
        $gid[] = $exploded[1];
      }
      else {
        $value = $object->get($item);

        if (!isset($value) || !strlen($value)) {
          return FALSE;
        }

        $gid[] = $value;
      }
    }

    return implode('-', $gid);
  }

  /**
   * Get the fixed mapping if it exists.
   *
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   *
   * @return array
   */
  public function getFixedMapping(ProfileInterface $profile): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function hasAttributes(): bool {
    return TRUE;
  }

  /**
   * @param \Drupal\a12s_maps_sync\Entity\ProfileInterface $profile
   * @param array $datum
   *
   * @return array
   */
  public static function generateAttributes(ProfileInterface $profile, array $datum): array {
    // Add the attributes.
    $attributes = [];
    foreach ($datum['attributes'] as $attributeCode => $attributeData) {
      $attribute = new Attribute($attributeCode);

      // If only one value, so it is a non-translatable / non-translated
      // attribute. In this case, MaPS will probably send a wrong idlanguage.
      // So we force it to the default one.
      if (count($attributeData['values']) === 1) {
        $attributeData['values'] = [$profile->getDefaultMapslanguage() => reset($attributeData['values'])];
      }

      $attribute->setData($attributeData['data'] ?? []);

      foreach ($attributeData['values'] as $idLanguage => $attributeDataByLang) {
        foreach ($attributeDataByLang as $delta => $attributeDatumByLang) {
          $attribute->setValue($attributeDatumByLang, $idLanguage, $delta);
        }
      }

      $attributes[$attributeCode] = $attribute;
    }

    return $attributes;
  }

  /**
   * {@inheritdoc}
   */
  public function findObjectFromEntity(ConverterInterface $converter, EntityInterface $entity): ?BaseInterface {
    if ($condition = $this->getConditionFromGid($converter, $entity)) {
      $data = $this->getData($converter, $condition, 1);
      return !empty($data) ? reset($data) : NULL;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getConditionFromGid(ConverterInterface $converter, EntityInterface $entity): ?array {
    $gid = $converter->getGid();

    $i = 0;
    foreach ($gid as $part) {
      if (in_array($part, $this->getAllowedGidEntityKeys())) {
        $exploded = explode('-', $entity->get(BaseInterface::GID_FIELD)->value);
        return [$part => ['value' => $exploded[$i]]];
      }
      $i++;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function isAttributable(): bool {
    return TRUE;
  }

}
