<?php

namespace Drupal\a12s_maps_sync\Routing;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteProvider;
use Drupal\Core\Routing\RouteSubscriberBase;
use Drupal\Core\Routing\RoutingEvents;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Subscriber for Devel routes.
 */
class RouteSubscriber extends RouteSubscriberBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * @var \Drupal\Core\Routing\RouteProvider
   */
  protected RouteProvider $routeProvider;

  /**
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_manager
   * @param \Drupal\Core\Routing\RouteProvider $router_provider
   */
  public function __construct(EntityTypeManagerInterface $entity_manager, RouteProvider $router_provider) {
    $this->entityTypeManager = $entity_manager;
    $this->routeProvider = $router_provider;
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection): void {
    $allowedEntityTypes = \Drupal::config('a12s_maps_sync.settings')->get('allowed_entity_types');

    if (!empty($allowedEntityTypes)) {
      $entityTypes = array_filter($allowedEntityTypes);

      foreach ($this->entityTypeManager->getDefinitions() as $entityTypeId => $entityType) {
        if (in_array($entityTypeId, $entityTypes)) {
          if ($route = $this->getEntityLoadRoute($entityType)) {
            $collection->add("entity.$entityTypeId.reimport_entity", $route);
          }
        }
      }
    }
  }

  /**
   * Gets the entity load route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entityType
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getEntityLoadRoute(EntityTypeInterface $entityType): ?Route {
    if ($reimportEntity = $entityType->getLinkTemplate('reimport-entity')) {
      $route = (new Route($reimportEntity))
        ->addDefaults([
          '_form' => '\Drupal\a12s_maps_sync\Form\ReimportEntityForm',
          '_title_callback' => 'Drupal\a12s_maps_sync\Form\ReimportEntityForm::title',
        ])
        ->addRequirements([
          '_permission' => 'reimport a12s maps sync entities',
        ])
        ->setOption('_admin_route', TRUE)
        ->setOption('_a12s_entity_type_id', $entityType->id());

      if ($parameters = $this->getRouteParameters($entityType, 'edit-form')) {
        $route->setOption('parameters', $parameters);
      }

      return $route;
    }
    return NULL;
  }

  /**
   * Gets the route parameters from the template.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param string $link_template
   *   The link template.
   *
   * @return array[]
   *   A list of route of parameters.
   */
  protected function getRouteParameters(EntityTypeInterface $entity_type, string $link_template): array {
    $parameters = [];
    if (!$path = $entity_type->getLinkTemplate($link_template)) {
      return $parameters;
    }

    $original_route_parameters = [];
    $candidate_routes = $this->routeProvider->getRoutesByPattern($path);
    if ($candidate_routes->count()) {
      // Guess the best match. There could be more than one route sharing the
      // same path. Try first an educated guess based on the route name. If we
      // can't find one, pick-up the first from the list.
      $name = 'entity.' . $entity_type->id() . '.' . str_replace('-', '_', $link_template);
      if (!$original_route = $candidate_routes->get($name)) {
        $iterator = $candidate_routes->getIterator();
        $iterator->rewind();
        $original_route = $iterator->current();
      }
      $original_route_parameters = $original_route->getOption('parameters') ?? [];
    }

    if (preg_match_all('/{\w*}/', $path, $matches)) {
      foreach ($matches[0] as $match) {
        $match = str_replace(['{', '}'], '', $match);
        // This match has an original route parameter definition.
        if (isset($original_route_parameters[$match])) {
          $parameters[$match] = $original_route_parameters[$match];
        }
        // It could be an entity type?
        elseif ($this->entityTypeManager->hasDefinition($match)) {
          $parameters[$match] = ['type' => "entity:$match"];
        }
      }
    }

    return $parameters;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[RoutingEvents::ALTER] = ['onAlterRoutes', 100];
    return $events;
  }

}
