<?php

namespace Drupal\a12s_maps_sync;

use Drupal\Core\Database\Query\SelectInterface;

final class StateBatch {

  /**
   * The batch id.
   *
   * @var int
   */
  private int $batchId;

  /**
   * The profile id.
   *
   * @var string
   */
  private string $profile;

  /**
   * The converter id.
   *
   * @var string|null
   */
  private ?string $converter;

  /**
   * The created timestamp.
   *
   * @var int
   */
  private int $created;

  /**
   * The updated timestamp.
   *
   * @var int
   */
  private int $updated = 0;

  /**
   * The current queue item id.
   *
   * @var int|null
   */
  private ?int $itemId = NULL;

  /**
   * The count of iterations.
   *
   * @var int
   */
  private int $itemIterationCount = 0;

  /**
   * @param int $batchId
   * @param string $profile
   * @param string|null $converter
   */
  public function __construct(int $batchId, string $profile, ?string $converter = NULL) {
    $this->batchId = $batchId;
    $this->profile = $profile;
    $this->converter = $converter;
    $this->created = time();
  }

  /**
   * @return int
   */
  public function getBatchId(): int {
    return $this->batchId;
  }

  /**
   * @return string
   */
  public function getProfile(): string {
    return $this->profile;
  }

  /**
   * @return string|null
   */
  public function getConverter(): ?string {
    return $this->converter;
  }

  /**
   * @return int
   */
  public function getCreated(): int {
    return $this->created;
  }

  /**
   * @return int
   */
  public function getUpdated(): int {
    return $this->updated;
  }

  /**
   * @return int|null
   */
  public function getItemId(): ?int {
    return $this->itemId;
  }

  /**
   * @return int
   */
  public function getItemIterationCount(): int {
    return $this->itemIterationCount;
  }

  /**
   * @return $this
   */
  public function resetItemIterationCount(): StateBatch {
    $this->itemIterationCount = 0;
    return $this;
  }

  /**
   * @return $this
   */
  public function incrementItemIterationCount(): StateBatch {
    $this->itemIterationCount++;
    return $this;
  }

  /**
   * @return $this
   */
  public function updateUpdated(): StateBatch {
    $this->updated = time();
    return $this;
  }

  /**
   * @param int|null $itemId
   *
   * @return StateBatch
   */
  public function setItemId(?int $itemId): StateBatch {
    $this->itemId = $itemId;
    return $this;
  }

  /**
   * Convert to an array.
   * Useful for displaying or debugging.
   *
   * @return string[]
   */
  public function toArray(): array {
    return [
      'batch_id' => $this->batchId,
      'profile' => $this->profile,
      'converter' => $this->converter,
      'created' => date('Y-m-d H:i:s', $this->created),
      'updated' => date('Y-m-d H:i:s', $this->updated),
      'item_id' => $this->itemId,
      'item_iteration_count' => $this->itemIterationCount,
    ];
  }

  /**
   * Get the remaining items in the Drupal queue.
   *
   * @return array
   * @throws \Exception
   */
  public function getRemainingItems(): array {
    return $this->getRemainingItemQuery()
      ->execute()
      ->fetchCol();
  }

  /**
   * Get the number remaining items in the Drupal queue.
   *
   * @return int
   * @throws \Exception
   */
  public function getRemainingItemsCount(): int {
    return (int) $this->getRemainingItemQuery()
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Query to get the remaining items in the Drupal queue.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   */
  private function getRemainingItemQuery(): SelectInterface {
    return \Drupal::database()->select('queue', 'q')
      ->fields('q', ['item_id'])
      ->condition('name', "drupal_batch:{$this->getBatchId()}:%", 'LIKE')
      ->range(0, 1);
  }

}
