<?php

namespace Drupal\a12s_maps_sync;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverter;

class BatchService {

  /**
   * Release the given lock.
   *
   * @param string $lockName
   */
  public static function releaseLock(string $lockName, &$context): void {
    $context['message'] = t('Purge lock');
    \Drupal::state()->delete($lockName);
  }

  /**
   * Process the import of the converter.
   *
   * @param \Drupal\a12s_maps_sync\Entity\MapsSyncConverter $converter
   * @param $limit
   * @param $context
   */
  public static function processConverterImport(MapsSyncConverter $converter, $limit, $step, $total, &$context) {
    $context['message'] = t('Importing converter @converter (@step / @total)',
      [
        '@converter' => $converter->label(),
        '@step' => $step,
        '@total' => $total,
      ]
    );

    $results = $converter->import($limit);

    $context['converter'] = $converter->id();
    $context['results'][] = $results;
  }

  /**
   * Batch Finished callback.
   *
   * @param bool $success
   *   Success of the operation.
   * @param array $results
   *   Array of results for post processing.
   * @param array $operations
   *   Array of operations.
   */
  public static function processConverterImportFinished($success, array $results, array $operations) {
    $messenger = \Drupal::messenger();

    if ($success) {
      if (!empty($results)) {
        $count = 0;
        foreach ($results as $result) {
          $count += count($result);
        }
        $messenger->addMessage(t('Import finished: @count results processed.', ['@count' => $count]));
      }
      else {
        $messenger->addMessage(t('No element processed. Maybe run the python script first, or check your converter name/configuration.'));
      }
    }
    else {
      // An error occurred.
      // $operations contains the operations that remained unprocessed.
      $error_operation = reset($operations);
      $messenger->addError(
        t('An error occurred while processing @operation with arguments : @args',
          [
            '@operation' => $error_operation[0],
            '@args' => print_r($error_operation[0], TRUE),
          ]
        )
      );
    }
  }

}
