<?php

namespace Drupal\a12s_maps_sync\Commands;

use Drupal\a12s_maps_sync\BatchService;
use Drupal\a12s_maps_sync\Entity\MapsSyncConverter;
use Drupal\a12s_maps_sync\Entity\MapsSyncProfile;
use Drupal\Core\Entity\EntityStorageException;
use Drush\Commands\DrushCommands;

/**
 * A Drush commandfile for a12s maps sync.
 */
class A12sMapsSyncCommands extends DrushCommands {

  /**
   * Rollback the given profile.
   *
   * @param string $profile
   *   The profile name.
   *
   * @command a12s_maps_sync:rollback:profile
   * @aliases amsrp
   *
   * @usage a12s_maps_sync:rollback:profile my_profile
   */
  public function rollbackProfile(string $profile) {
    $messenger = \Drupal::messenger();

    // Load the profile.
    try {
      $profile = MapsSyncProfile::load($profile);

      $messenger->addMessage('Rollback profile ' . $profile->label());

      /** @var \Drupal\a12s_maps_sync\Entity\MapsSyncConverter $converter */
      foreach ($profile->getConverters() as $converter) {
        $messenger->addMessage("-- Rollback converter {$converter->label()}");

        $results = $converter->rollback();

        $messenger->addStatus('-- ' . count($results) . ' elements deleted');
      }
    }
    catch (EntityStorageException $e) {
      $messenger->addError("The profile $profile does not exist");
      return;
    }

    $messenger->addStatus('Rollback finished');
  }

  /**
   * Rollback the given converter.
   *
   * @param string $converter
   *   The converter name.
   *
   * @command a12s_maps_sync:rollback:converter
   * @aliases amsrc
   *
   * @usage a12s_maps_sync:rollback:converter my_converter
   */
  public function rollbackConverter(string $converter) {
    $results = [];

    $messenger = \Drupal::messenger();

    // Try to load the converter.
    try {
      $converter = MapsSyncConverter::load($converter);
      $messenger->addMessage('Rollback converter ' . $converter->label());

      if ($converter !== NULL) {
        $results = $converter->rollback();
      }
    } catch (EntityStorageException $e) {
      $messenger->addError($e->getMessage());
    }

    if (!empty($results)) {
      $messenger->addStatus('Rollback finished');
      $messenger->addStatus(count($results) . ' elements deleted');
    }
  }

  /**
   * Import the given profile.
   *
   * @param string $profile
   *   The profile name.
   * @param array $options
   *   The options array.
   * @option int limit
   *   The limit of elements to process.
   * @option bool force
   *   Force the import despite the lock status.
   *
   * @command a12s_maps_sync:import:profile
   * @aliases amsip
   *
   * @usage a12s_maps_sync:import:profile my_profile --limit=20 --force=true
   */
  public function importProfile(string $profile, $options = ['force' => 'false']) {
    // Load the profile.
    try {
      $profile = MapsSyncProfile::load($profile);

      $lockName = 'a12s_maps_sync:lock:profile:' . $profile->id();
      $operations = [];

      $config = \Drupal::config('a12s_maps_sync.settings');

      if ((bool) $config->get('ignore_lock') === TRUE || strtolower((string) $options['force']) === 'true' || !\Drupal::state()->get($lockName)) {
        // Set the lock.
        \Drupal::state()->set($lockName, time());

        $limit = $options['limit'] ?? $config->get('import_batch_size');

        /** @var \Drupal\a12s_maps_sync\Entity\MapsSyncConverter $converter */
        foreach ($profile->getConverters() as $converter) {
          // Get the total of elements.
          $mapsManager = $converter->getMapsManager();
          $maps_sync_db = $mapsManager->getConnection(TRUE);

          $count = $mapsManager->queueBasesCountQuery($maps_sync_db, $converter);
          $steps = (int) ceil($count / $limit);

          for ($i = 1; $i <= $steps; $i++) {
            $operations[] = [BatchService::class . '::processConverterImport', [$converter, $limit, $i, $steps]];
          }
        }

        $operations[] = [BatchService::class . '::releaseLock', [$lockName]];

        $batch = [
          'title' => t('Processing profile @profile import', ['@profile' => $profile->label()]),
          'operations' => $operations,
          'finished' => BatchService::class . '::processConverterImportFinished',
        ];

        batch_set($batch);
        drush_backend_batch_process();
      }
      else {
        $this->logger()->error("The {$profile->id()} profile is locked. You can remove the lock with the following command: drush amsrl {$profile->id()}");
      }
    }
    catch (EntityStorageException $e) {
      $this->logger()->error("The profile $profile does not exist");
      exit(500);
    }
  }

  /**
   * Import the given converter.
   *
   * @param string $converter
   *   The converter name.
   * @param array $options
   *   The options array.
   * @option int limit
   *   The limit of elements to process.
   * @option bool force
   *   Force the import despite the lock status.
   *
   * @command a12s_maps_sync:import:converter
   * @aliases amsic
   *
   * @usage a12s_maps_sync:import:converter my_converter --limit=20 --force=true
   */
  public function importConverter(string $converter, $options = ['force' => 'false']) {
    /** @var \Drupal\a12s_maps_sync\MapsSyncConverterManager $converter_manager */
    $converter = MapsSyncConverter::load($converter);

    $profile = $converter->getProfile();
    $lockName = 'a12s_maps_sync:lock:profile:' . $profile->id();

    $config = \Drupal::config('a12s_maps_sync.settings');

    if ((bool) $config->get('ignore_lock') === TRUE || strtolower((string) $options['force']) === 'true' || !\Drupal::state()->get($lockName)) {
      // Set the lock.
      \Drupal::state()->set($lockName, time());

      $limit = $options['limit'] ?? $config->get('import_batch_size');

      // Try to load the converter.
      try {
        if ($converter !== NULL) {
          // Get the total of elements.
          $mapsManager = $converter->getMapsManager();
          $maps_sync_db = $mapsManager->getConnection(TRUE);

          $count = $mapsManager->queueBasesCountQuery($maps_sync_db, $converter);
          $steps = (int) ceil($count / $limit);

          $batch = [
            'title' => t('Processing converter @converter import', ['@converter' => $converter->label()]),
            'operations' => [],
            'finished' => BatchService::class . '::processConverterImportFinished',
          ];

          for ($i = 1; $i <= $steps; $i++) {
            $batch['operations'][] = [BatchService::class . '::processConverterImport', [$converter, $limit, $i, $steps]];
          }

          $batch['operations'][] = [BatchService::class . '::releaseLock', [$lockName]];

          batch_set($batch);
          drush_backend_batch_process();
        }
      } catch (\Exception $e) {
        $this->logger()->error($e->getMessage());
        exit(500);
      }
    }
    else {
      $this->logger()->error("The {$profile->id()} profile is locked. You can remove the lock with the following command: drush amsrl {$profile->id()}");
    }
  }

  /**
   * @param string $profile
   *   The profile machine name.
   *
   * @command a12s_maps_sync:release_lock
   * @aliases amsrl
   *
   * @usage a12s_maps_sync:release_lock my_converter
   */
  public function releaseLock(string $profile) {
    $this->output()->writeln('Releasing lock for profile ' . $profile);

    \Drupal::state()->delete('a12s_maps_sync:lock:profile:' . $profile);

    $this->logger()->success('Lock released');
  }

}
