<?php

namespace Drupal\a12s_maps_sync\Entity;

use Drupal\a12s_maps_sync\Event\ConverterItemImportEvent;
use Drupal\a12s_maps_sync\Maps\MapsBaseInterface;
use Drupal\a12s_maps_sync\Maps\MapsBaseManagerInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Utility\Error;
use Drupal\language\Entity\ContentLanguageSettings;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Defines the Maps sync converter entity.
 *
 * @ConfigEntityType(
 *   id = "maps_sync_converter",
 *   label = @Translation("MaPS Sync Converter"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\a12s_maps_sync\MapsSyncConverterListBuilder",
 *     "form" = {
 *       "add" = "Drupal\a12s_maps_sync\Form\MapsSyncConverterForm",
 *       "edit" = "Drupal\a12s_maps_sync\Form\MapsSyncConverterForm",
 *       "delete" = "Drupal\a12s_maps_sync\Form\MapsSyncConverterDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\a12s_maps_sync\MapsSyncConverterHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "maps_sync_converter",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "uuid",
 *     "langcode",
 *     "status",
 *     "dependencies",
 *     "id",
 *     "label",
 *     "profile_id",
 *     "entity_type",
 *     "bundle",
 *     "filters",
 *     "mapping",
 *     "published_statuses",
 *     "status_management",
 *     "gid",
 *     "maps_type",
 *     "handler_id",
 *     "parent_id",
 *     "weight",
 *     "media_status_published_value",
 *     "media_status_property"
 *   },
 *   links = {
 *     "canonical" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}",
 *     "add-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/add",
 *     "edit-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/edit",
 *     "delete-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter/{maps_sync_converter}/delete",
 *     "collection" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/converter"
 *   }
 * )
 */
class MapsSyncConverter extends ConfigEntityBase implements MapsSyncConverterInterface {

  /**
   * The Maps sync converter ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The Maps sync converter label.
   *
   * @var string
   */
  protected $label;

  /**
   * {@inheritdoc}
   */
  public function getProfile(): ?MapsSyncProfileInterface {
    if ($this->get('profile_id') === NULL) {
      return NULL;
    }
    return MapsSyncProfile::load($this->get('profile_id'));
  }

  /**
   * {@inheritdoc}
   */
  public function setProfileId(string $profile_id): MapsSyncConverterInterface {
    $this->set('profile_id', $profile_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConverterEntityType(): ?string {
    return $this->get('entity_type');
  }

  /**
   * {@inheritdoc}
   */
  public function setConverterEntityType(string $entity_type): MapsSyncConverterInterface {
    $this->set('entity_type', $entity_type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConverterBundle(): ?string {
    return $this->get('bundle');
  }

  /**
   * {@inheritdoc}
   */
  public function setConverterBundle(string $bundle): MapsSyncConverterInterface {
    $this->set('bundle', $bundle);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFilters(): ?array {
    return $this->get('filters');
  }

  /**
   * {@inheritdoc}
   */
  public function setFilters(array $filters): MapsSyncConverterInterface {
    $this->set('filters', $filters);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMapping(): array {
    $mapping = $this->get('mapping');
    return !empty($mapping) ? $mapping : [];
  }

  /**
   * {@inheritdoc}
   */
  public function setMapping(array $mapping): MapsSyncConverterInterface {
    $this->set('mapping', $mapping);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPublishedStatuses(): array {
    return $this->get('published_statuses') ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function setPublishedStatuses(array $statuses): MapsSyncConverterInterface {
    $this->set('published_statuses', $statuses);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusManagement(): ?string {
    return $this->get('status_management');
  }

  /**
   * {@inheritdoc}
   */
  public function getMediaStatusProperty(): ?string {
    return $this->get('media_status_property');
  }

  /**
   * {@inheritdoc}
   */
  public function setMediaStatusProperty(string $mediaStatusProperty): MapsSyncConverterInterface {
    $this->set('media_status_property', $mediaStatusProperty);
    return $this;
  }

  /**
   * @inheritDoc
   */
  public function getMediaStatusPublishedValue(): ?string {
    return $this->get('media_status_published_value');
  }

  /**
   * @inheritDoc
   */
  public function setMediaStatusPublishedValue(string $publishedValue): MapsSyncConverterInterface {
    $this->set('media_status_published_value', $publishedValue);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setStatusManagement(string $statusManagement): MapsSyncConverterInterface {
    if (!in_array($statusManagement, [
      MapsSyncConverterInterface::STATUS_MANAGEMENT_DELETE,
      MapsSyncConverterInterface::STATUS_MANAGEMENT_UNPUBLISH
    ], TRUE)) {
      throw new \InvalidArgumentException('The given status management method is not valid.');
    }

    $this->set('status_management', $statusManagement);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getGid(): array {
    return $this->get('gid') ?: [];
  }

  /**
   * {@inheritdoc}
   */
  public function setGid(array $gid): MapsSyncConverterInterface {
    $this->set('gid', $gid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMapsType(): ?string {
    return $this->get('maps_type');
  }

  /**
   * {@inheritdoc}
   */
  public function setMapsType(string $type): MapsSyncConverterInterface {
    $this->set('maps_type', $type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMapsManager(): MapsBaseManagerInterface {
    switch ($this->getMapsType()) {
      case 'library':
        $service = 'a12s_maps_sync.maps_library_manager';
        break;
      case 'media':
        $service = 'a12s_maps_sync.maps_media_manager';
        break;
      default:
        $service = 'a12s_maps_sync.maps_object_manager';
        break;
    }

    return \Drupal::service($service);
  }

  /**
   * {@inheritdoc}
   */
  public function getHandler(): MapsSyncHandlerInterface {
    $plugins = \Drupal::service('plugin.manager.maps_sync_handler');
    return $plugins->createInstance($this->getHandlerId());
  }

  /**
   * {@inheritdoc}
   */
  public function getHandlerId(): ?string {
    return $this->get('handler_id');
  }

  /**
   * {@inheritdoc}
   */
  public function setHandlerId(string $handler_id): MapsSyncConverterInterface {
    $this->set('handler_id', $handler_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getParentId(): ?string {
    return $this->get('parent_id');
  }

  /**
   * {@inheritdoc}
   */
  public function setParentId(string $parent_id): MapsSyncConverterInterface {
    $this->set('parent_id', $parent_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getParent(): ?MapsSyncConverterInterface {
    return self::load($this->get('parent_id'));
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): ?int {
    return $this->get('weight');
  }

  /**
   * {@inheritdoc}
   */
  public function setWeight(int $weight): MapsSyncConverterInterface {
    $this->set('weight', $weight);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function import($limit): array {
    $results = [];

    /** @var EventDispatcherInterface $event_dispatcher */
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $mapsManager = $this->getMapsManager();
    $maps_sync_db = $mapsManager->getConnection(TRUE);
    $queue_items = $mapsManager->getBaseFromQueue($maps_sync_db, $this, $limit);

    foreach ($queue_items as $queue) {
      $object_ids = [];

      foreach ($queue as $item) {
        $object_ids[$item['id']] = $item['data_id'];
      }

      $objects = $mapsManager->buildBases($this, $object_ids);

      foreach ($objects as $object) {
        $result = [
          'object' => $object,
          'converter' => $this,
          'profile' => $this->getProfile(),
          'entity_type' => $this->getConverterEntityType(),
          'entity_id' => NULL,
          'success' => FALSE,
          'gid' => NULL,
        ];

        //if ($this->converterApplies($object, $settings)) {
        //  $result['converter'] = $alias;
        //  break;
        //}

        // Check if the entity is translatable.
        $config = ContentLanguageSettings::loadByEntityTypeBundle($this->getConverterEntityType(), $this->getConverterBundle());

        $default_language = \Drupal::languageManager()->getDefaultLanguage();
        if ($config->getThirdPartySetting('content_translation', 'enabled')) {
          $languages = \Drupal::languageManager()->getLanguages();

          // Set the default language first.
          unset($languages[$default_language->getId()]);

          $languages = array_values($languages);
          array_unshift($languages, $default_language);
        }
        else {
          $languages = [NULL];
        }

        foreach ($languages as $language) {
          try {
            $entity = $this->getHandler()->convertItem($object, $this, $language);

            // Entity is null if entity has been deleted or skipped.
            // @todo Check if logic is correct.
            if ($entity === NULL) {
              $queue_id = array_search($object->getId(), $object_ids);
              $mapsManager->removeFromQueue($maps_sync_db, $queue_id);
              continue;
            }

            $result['gid'] = $entity->get(MapsBaseInterface::GID_FIELD)->value;

            if ($entity) {
              // Add context to the entity for contributed modules and allow for
              // modifications of the $result array.
              $entity->context['a12s_maps_sync'] = &$result;

              $this->getHandler()->preSave($entity, $object, $this, $language);
              $result['success'] = $entity->save();
              $this->getHandler()->postSave($entity, $object, $this, $language);

              $result['entity_id'] = $entity->id();

              $event = new ConverterItemImportEvent($this, $object, $entity, $language);
              $event_dispatcher->dispatch($event, ConverterItemImportEvent::FINISHED);
            }
          }
          catch (\Exception $e) {
            Error::logException(\Drupal::logger('a12s_maps_sync'), $e);
          }
          finally {
            $results[] = $result;

            // @todo handle retrial on failure?

            $queue_id = array_search($object->getId(), $object_ids);
            $mapsManager->removeFromQueue($maps_sync_db, $queue_id);
          }

          // @todo Such process means that with  several converters using the same
          // profile, only the first converter will proceed.
          // We need to handle the filters to have a correct behaviour.
          // @todo Handle defined filters (see commented code above).
        }
      }
    }
    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function rollback() {
    $entity_type = \Drupal::entityTypeManager()->getDefinition($this->getConverterEntityType());

    $query = \Drupal::entityQuery($this->getConverterEntityType())->accessCheck(FALSE);

    $bundle = $this->getConverterBundle();
    if ($bundle !== $entity_type->id()) {
      $query->condition($entity_type->getKey('bundle'), $bundle);
    }

    $deleted = [];

    $results = $query->execute();
    foreach ($results as $id) {
      $entity = \Drupal::entityTypeManager()->getStorage($entity_type->id())->load($id);

      if ($entity !== NULL && $entity->hasField(MapsBaseInterface::GID_FIELD) && $entity->get(MapsBaseInterface::GID_FIELD)->value) {
        try {
          $entity->delete();
          $deleted[] = $id;
        }
        catch (EntityStorageException $e) {
          Error::logException(\Drupal::logger('a12s_maps_sync'), $e);
        }
      }
    }

    return $deleted;
  }

  /**
   * {@inheritdoc}
   */
  protected function urlRouteParameters($rel) {
    $uri_route_parameters = parent::urlRouteParameters($rel);
    $uri_route_parameters['maps_sync_profile'] = $this->getProfile()->id();
    return $uri_route_parameters;
  }

}
