<?php

namespace Drupal\a12s_maps_sync\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the MaPS Sync profile entity.
 *
 * @ConfigEntityType(
 *   id = "maps_sync_profile",
 *   label = @Translation("MaPS Sync Profile"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\a12s_maps_sync\MapsSyncProfileListBuilder",
 *     "form" = {
 *       "add" = "Drupal\a12s_maps_sync\Form\MapsSyncProfileForm",
 *       "edit" = "Drupal\a12s_maps_sync\Form\MapsSyncProfileForm",
 *       "delete" = "Drupal\a12s_maps_sync\Form\MapsSyncProfileDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\a12s_maps_sync\MapsSyncProfileHtmlRouteProvider",
 *     },
 *   },
 *   config_prefix = "maps_sync_profile",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "uuid",
 *     "langcode",
 *     "status",
 *     "dependencies",
 *     "id",
 *     "label",
 *     "default_aps_language",
 *     "medias_pathm"
 *   },
 *   links = {
 *     "canonical" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}",
 *     "add-form" = "/admin/a12s_maps_sync/profile/add",
 *     "edit-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/edit",
 *     "delete-form" = "/admin/a12s_maps_sync/profile/{maps_sync_profile}/delete",
 *     "collection" = "/admin/a12s_maps_sync/profile",
 *   }
 * )
 */
class MapsSyncProfile extends ConfigEntityBase implements MapsSyncProfileInterface {

  /**
   * The MaPS Sync profile ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The MaPS Sync profile label.
   *
   * @var string
   */
  protected $label;

  /**
   * The default MaPS System language id.
   */
  protected $default_maps_language;

  /**
   * The MaPS medias path.
   */
  protected $medias_path;

  /**
   * {@inheritdoc}
   */
  public function getConverters(): array {
    $query = \Drupal::service('entity_type.manager')->getStorage('maps_sync_converter')->getQuery();
    $query
      ->condition('profile_id', $this->id())
      ->sort('weight', 'ASC');

    $ids = $query->execute();
    return MapsSyncConverter::loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultMapsLanguage(): ?int {
    return $this->default_maps_language;
  }

  /**
   * {@inheritdoc}
   */
  public function setDefaultMapsLanguage(int $language): MapsSyncProfileInterface {
    $this->default_maps_language = $language;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMediasPath(): ?string {
    return $this->medias_path;
  }

  /**
   * {@inheritdoc}
   */
  public function setMediasPath(string $medias_path): MapsSyncProfileInterface {
    $this->medias_path = $medias_path;
    return $this;
  }

}
