<?php

namespace Drupal\a12s_maps_sync\Form;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Entity\MapsSyncProfileInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Serialization\Yaml;

/**
 * Class MapsSyncConverterForm.
 */
class MapsSyncConverterForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state, MapsSyncProfileInterface $maps_sync_profile = NULL) {
    $form = parent::form($form, $form_state);

    if ($maps_sync_profile === NULL) {
      $maps_sync_profile = \Drupal::request()->get('maps_sync_profile');
    }

    /** @var \Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface $converter */
    $converter = $this->entity;
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $converter->label(),
      '#description' => $this->t('Label for the Maps sync converter.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $converter->id(),
      '#machine_name' => [
        'exists' => '\Drupal\a12s_maps_sync\Entity\MapsSyncConverter::load',
      ],
      '#disabled' => !$converter->isNew(),
    ];

    $form['profile_id'] = [
      '#type' => 'value',
      '#value' => $maps_sync_profile->id(),
    ];

    $plugins = \Drupal::service('plugin.manager.maps_sync_handler');
    $plugin_definitions = $plugins->getDefinitions();
    $options = [];
    foreach ($plugin_definitions as $plugin_id => $plugin_definition) {
      $options[$plugin_id] = $plugin_definition['label'];
    }

    $form['handler_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Handler'),
      '#options' => $options,
      '#default_value' => $converter->getHandlerId() ?: 'default',
      '#required' => TRUE,
    ];

    $form['gid'] = [
      '#type' => 'textfield',
      '#title' => $this->t('GID'),
      '#description' => $this->t('GIDs, separated by commas'),
      '#default_value' => $converter->getGid() ? implode(',', $converter->getGid()) : 'profile,source_id',
      '#required' => TRUE,
      '#multiple' => TRUE,
    ];

    $entity_types = array_filter(\Drupal::config('a12s_maps_sync.settings')->get('allowed_entity_types'));

    $options = [];
    foreach ($entity_types as $entity_type_id) {
      $entity_type = \Drupal::entityTypeManager()->getDefinition($entity_type_id);
      $options[$entity_type->id()] = $entity_type->getLabel();
    }
    asort($options);

    $form['entity_type'] = [
      '#type' => 'select',
      '#options' => $options,
      '#title' => $this->t('Entity type'),
      '#default_value' => $converter->getConverterEntityType(),
      '#required' => TRUE,
    ];

    foreach ($entity_types as $entity_type_id) {
      $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo($entity_type_id);

      if (!empty($bundles)) {
        $options = [];
        foreach ($bundles as $bundle_id => $bundle) {
          $options[$bundle_id] = $bundle['label'];
        }
        asort($options);

        $form['bundle_' . $entity_type_id] = [
          '#type' => 'select',
          '#options' => $options,
          '#title' => $this->t('Bundle'),
          '#default_value' => $converter->getConverterBundle(),
          '#states' => [
            'visible' => [
              ':input[name="entity_type"]' => ['value' => $entity_type_id],
            ],
          ],
        ];
      }
    }

    $form['maps_type'] = [
      '#type' => 'select',
      '#title' => $this->t('MaPS type'),
      '#default_value' => $converter->getMapsType(),
      '#required' => TRUE,
      '#options' => [
        'object' => $this->t('Object'),
        'media' => $this->t('Media'),
        'library' => $this->t('Library'),
      ],
    ];

    $form['published_statuses'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Published statuses'),
      '#description' => $this->t('Published statuses ids, separated by commas'),
      '#default_value' => $converter->getPublishedStatuses() ? implode(',', $converter->getPublishedStatuses()) : '',
      '#multiple' => TRUE,
      '#states' => [
        'visible' => [
          'select[name="maps_type"]' => ['value' => 'object'],
        ],
      ],
    ];

    $form['status_management'] = [
      '#type' => 'select',
      '#title' => $this->t('Status management'),
      '#description' => $this->t('Defines how the unpublished statuses are managed.'),
      '#default_value' => $converter->getStatusManagement() ?? '',
      '#options' => [
        MapsSyncConverterInterface::STATUS_MANAGEMENT_UNPUBLISH => $this->t('Unpublish the entity (if possible)'),
        MapsSyncConverterInterface::STATUS_MANAGEMENT_DELETE => $this->t('Delete the entity'),
        MapsSyncConverterInterface::STATUS_MANAGEMENT_UPDATE_EXISTING_IGNORE_STATUS => $this->t('Update the entity but keep the status unchanged'),
      ],
      '#empty_option' => $this->t('- Select -'),
    ];

    $form['media_status_property'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Status property'),
      '#description' => $this->t("Defines the property used for the media’s status"),
      '#states' => [
        'visible' => [
          'select[name="status_management"]' => ['value' => MapsSyncConverterInterface::STATUS_MANAGEMENT_DELETE],
          'select[name="maps_type"]' => ['value' => 'media'],
        ],
      ],
      '#default_value' => $converter->getMediaStatusProperty() ?? '',
    ];

    $form['media_status_published_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Status published value'),
      '#description' => $this->t("Defines the value used for the media’s published status"),
      '#states' => [
        'visible' => [
          'select[name="status_management"]' => ['value' => MapsSyncConverterInterface::STATUS_MANAGEMENT_DELETE],
          'select[name="maps_type"]' => ['value' => 'media'],
        ],
      ],
      '#default_value' => $converter->getMediaStatusPublishedValue() ?? '',
    ];

    $form['filters'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Filters'),
      '#description' => $this->t("The converter's filters, in a Yaml format."),
      '#default_value' => !empty($converter->getFilters()) ? Yaml::encode($converter->getFilters()) : '',
    ];

    $form['mapping'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Mapping'),
      '#description' => $this->t("The converter's mapping, in a Yaml format."),
      '#default_value' => !empty($converter->getMapping()) ? Yaml::encode($converter->getMapping()) : '',
    ];

    return $form;
  }

  /**
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    $json_fields = ['filters', 'mapping'];
    foreach ($json_fields as $json_field) {
      if (!empty($values[$json_field]) && !is_array(Yaml::decode($values[$json_field]))) {
        $form_state->setErrorByName($json_field);
      }
    }

    // Only keep the needed bundle (because of Form states).
    $bundle_key = 'bundle_' . $form_state->getValue('entity_type');
    if ($bundle = $form_state->getValue($bundle_key)) {
      $values['bundle'] = $bundle;
    }

    // Some post process.
    $values['published_statuses'] = explode(',', $values['published_statuses']);
    $values['gid'] = explode(',', $values['gid']);
    $values['mapping'] = Yaml::decode($values['mapping']);
    $values['filters'] = Yaml::decode($values['filters']);

    foreach ($values as $key => $value) {
      // Remove useless bundles.
      if (strpos($key, 'bundle_') !== 0) {
        $entity->set($key, $value);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $maps_sync_converter = $this->entity;
    $status = $maps_sync_converter->save();

    switch ($status) {
      case SAVED_NEW:
        \Drupal::messenger()->addMessage($this->t('Created the %label Maps sync converter.', [
          '%label' => $maps_sync_converter->label(),
        ]));
        break;

      default:
        \Drupal::messenger()->addMessage($this->t('Saved the %label Maps sync converter.', [
          '%label' => $maps_sync_converter->label(),
        ]));
    }
    $form_state->setRedirectUrl($maps_sync_converter->toUrl('collection'));
  }

}
