<?php

namespace Drupal\a12s_maps_sync\Maps;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Entity\MapsSyncProfileInterface;
use Drupal\a12s_maps_sync\Maps\Exception\MapsEntityDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Drupal\Core\Database\DatabaseNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Utility\Error;
use Psr\Log\LoggerInterface;

abstract class MapsBaseManager implements MapsBaseManagerInterface {

  /**
   * The MaPS database name. It needs to be defined in the database info, either
   * in the settings.php file or using a hook.
   */
  const DATABASE_TARGET = 'a12s_maps_sync';

  /**
   * The character used to split each parts of the global identifier.
   */
  const GID_SEPARATOR = '-';

  /**
   * The connection to MaPS Sync database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The data type used in the python table.
   *
   * @var string
   */
  protected $data_type;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The a12s_maps_sync logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public function queueBasesBaseQuery(Connection $database, MapsSyncConverterInterface $converter) {
    $query = $database->select('queue', 'q')
      ->condition('q.profile', $converter->getProfile()->id())
      ->condition('q.data_type', $this->data_type);

    // Check if there is any filters, and manage them.
    foreach ($converter->getFilters() as $filter_name => $filter_value) {
      if (strpos($filter_name, 'converter') === 0) {
        $query->condition('q.converter', $filter_value);
      }
      else {
        if (strpos($filter_name, 'object_class') === 0) {
          $query->join('object_classes', 'oc', "oc.object_id = q.data_id AND class_id = $filter_value");
        }
        else {
          if (strpos($filter_name, 'object_') === 0) {
            $filter_name = str_replace('object_', '', $filter_name);
            $query->join('objects', 'o', "o.id = q.data_id AND $filter_name = '$filter_value'");
          }
          // Attribute filter.
          else {
            // @todo work only for medias, so move this in the media class.
            if ($this->data_type === 'media') {
              // Add a junction with the medias table.
              if (!isset($query->getTables()['m'])) {
                $query->join('medias', 'm', "m.id = q.data_id");
              }

              if (strpos($filter_name, 'attribute_') === 0) {
                MapsMediaManager::addConditionOnMediaAttribute($database, $query, $filter_name, $filter_value);
              }
              else {
                if (strpos($filter_name, 'media_') === 0) {
                  MapsMediaManager::addConditionOnMediaProperty($query, $filter_name, $filter_value);
                }
              }
            }
          }
        }
      }
    }

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function queueBasesCountQuery(Connection $database, MapsSyncConverterInterface $converter): int {
    return $this->queueBasesBaseQuery($database, $converter)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The a12s_maps_sync logger channel.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, LoggerInterface $logger) {
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public function getConnection($throw_error = FALSE) {
    if (!isset($this->connection)) {
      if (!Database::getConnectionInfo(static::DATABASE_TARGET)) {
        $message = 'The connection to MaPS Sync database is missing';

        if ($throw_error) {
          throw new DatabaseNotFoundException($message);
        }
        else {
          \Drupal::logger('a12s_maps_sync')->error($message);
        }

        return NULL;
      }

      $this->connection = Database::getConnection('default', static::DATABASE_TARGET);
    }

    return $this->connection;
  }

  /**
   * {@inheritdoc}
   */
  public function closeConnection() {
    if ($this->connection) {
      Database::closeConnection('default', static::DATABASE_TARGET);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function removeFromQueue(Connection $database, int $id): int {
    return $database->delete('queue')
      ->condition('id', $id)
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getEntity(string $entity_type, ?string $bundle, MapsBaseInterface $object, array $gid_definition, LanguageInterface $language = NULL) {
    try {
      if (empty($gid_definition)) {
        throw new MapsEntityDefinitionException($entity_type, 'gid');
      }

      $langcode = $language !== NULL ? $language->getId() : LanguageInterface::LANGCODE_NOT_SPECIFIED;

      // @todo how to handle object translatable attributes?
      if ($gid = $this->getGid($gid_definition, $object)) {
        $storage = $this->entityTypeManager->getStorage($entity_type);
        $entities = $storage->loadByProperties([MapsBaseInterface::GID_FIELD => $gid]);
        // @todo load translation?

        // @todo how to handle multiple results? This should never happen!
        if ($entities) {
          $entity = reset($entities);
          if (in_array($langcode, [
            \Drupal::languageManager()->getDefaultLanguage()->getId(),
            LanguageInterface::LANGCODE_NOT_SPECIFIED,
          ], TRUE)) {
            return $entity;
          }

          return $entity->hasTranslation($langcode) ? $entity->getTranslation($langcode) : $entity->addTranslation($langcode);
        }

        $time = \Drupal::time()->getRequestTime();
        $params = [
          MapsBaseInterface::GID_FIELD => $gid,
          // @todo Use $definition['translatable'] and force "und" langcode?
          'langcode' => $langcode,
          'created' => $time,
          'changed' => $time,
        ];

        if ($bundle !== NULL) {
          /** @var EntityTypeManagerInterface $entity_type_manager */
          $entity_type_manager = \Drupal::service('entity_type.manager');
          $type = $entity_type_manager->getDefinition($entity_type);

          $params[$type->getKey('bundle')] = $bundle;
        }

        return $storage->create($params);
      }
    }
    catch (PluginNotFoundException $e) {
      $this->logger->error('The entity type %entity_type is not defined.', ['%entity_type' => $entity_type]);
    }
    catch (\Exception $e) {
      Error::logException($this->logger, $e);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getGid(array $definition, MapsBaseInterface $object) {
    $gid = [];

    foreach ($definition as $item) {
      $exploded = explode(':', $item);

      // If the element begins with "const:", it is a constant, and we don't
      // have to get any value...
      if (count($exploded) === 2 && $exploded[0] === 'const') {
        $gid[] = $exploded[1];
      }
      else {
        $value = $object->get($item);

        if (!isset($value) || !strlen($value)) {
          return FALSE;
        }

        $gid[] = $value;
      }
    }

    return implode(static::GID_SEPARATOR, $gid);
  }

  /**
   * {@inheritdoc}
   */
  public function getGidFromArray(array $definition, array $array) {
    $gid = [];

    foreach ($definition as $item) {
      $exploded = explode(':', $item);

      // If the element begins with "const:", it is a constant, and we don't
      // have to get any value...
      if (count($exploded) === 2 && $exploded[0] === 'const') {
        $gid[] = $exploded[1];
      }
      else {
        if (empty($array[$item])) {
          return FALSE;
        }

        $gid[] = $array[$item];
      }
    }

    return implode(static::GID_SEPARATOR, $gid);
  }

  /**
   * Get the fixed mapping if it exists.
   *
   * @return array
   */
  public function getFixedMapping(MapsSyncProfileInterface $profile): array {
    return [];
  }

}
