<?php

namespace Drupal\a12s_maps_sync\Maps;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Language\LanguageInterface;

interface MapsBaseManagerInterface {

  /**
   * Get the database connection.
   *
   * @param bool $throw_error
   *   Whether to throw an exception if the database connection is missing.
   *
   * @return \Drupal\Core\Database\Connection|null
   */
  public function getConnection($throw_error = FALSE);

  /**
   * Close the connection to the MaPS Sync database.
   */
  public function closeConnection();

  /**
   * Delete the given item fromthe queue.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The "maps_sync" database connection.
   * @param int $id
   *   The queue item ID.
   *
   * @return int
   *   The number of rows affected by the delete query.
   */
  public function removeFromQueue(Connection $database, int $id): int;

  /**
   * Get the Drupal entity related to the given object.
   *
   * @param string $entity_type
   *   The target entity type.
   * @param string|null $bundle
   *   The target bundle.
   * @param \Drupal\a12s_maps_sync\Maps\MapsObjectInterface $object
   *   The MaPS Object.
   * @param array $gid_definition
   *   The GID definition.
   * @param \Drupal\Core\Language\LanguageInterface|null $language
   *
   * @return \Drupal\Core\Entity\EntityInterface|\Drupal\Core\Entity\FieldableEntityInterface|false
   */
  public function getEntity(string $entity_type, ?string $bundle, MapsBaseInterface $object, array $gid_definition, LanguageInterface $language = NULL);

  /**
   * Get the global identifier from the given definition and object.
   *
   * @param array $definition
   *   The GID definition.
   * @param \Drupal\a12s_maps_sync\Maps\MapsObjectInterface $object
   *   The related MaPS Object.
   *
   * @return string|FALSE
   *   The GID value or FALSE on failure.
   */
  public function getGid(array $definition, MapsBaseInterface $object);

  /**
   * Get the global identifier from the given definition and array.
   *
   * @param array $definition
   *   The GID definition.
   * @param array $array
   *   An array of values.
   *
   * @return string|FALSE
   *   The GID value or FALSE on failure.
   */
  public function getGidFromArray(array $definition, array $array);

  /**
   * Base query for objects in queue.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The "maps_sync" database connection.
   * @param MapsSyncConverterInterface $converter
   *   The converter.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   */
  public function queueBasesBaseQuery(Connection $database, MapsSyncConverterInterface $converter);

  /**
   * Count query for objects in queue.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The "maps_sync" database connection.
   * @param MapsSyncConverterInterface $converter
   *   The converter.
   *
   * @return \Drupal\Core\Database\Query\SelectInterface
   */
  public function queueBasesCountQuery(Connection $database, MapsSyncConverterInterface $converter): int;

  /**
   * Get a set of objects from the queue.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The "maps_sync" database connection.
   * @param MapsSyncConverterInterface $converter
   *   The converter.
   * @param int $limit
   *   The maximum number of objects to retrieve.
   *
   * @return array
   *   A list of object IDs, grouped by profile name.
   */
  public function getBaseFromQueue(Connection $database, MapsSyncConverterInterface $converter, $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array;

  /**
   * Build a set of MaPS Object instances.
   *
   * @param MapsSyncConverterInterface $converter
   *   The converter.
   * @param array $object_ids
   *   The list of object IDs.
   *
   * @return \Drupal\a12s_maps_sync\Maps\MapsObject[]
   */
  public function buildBases(MapsSyncConverterInterface $converter, array $object_ids): array;

}
