<?php

namespace Drupal\a12s_maps_sync\Maps;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\DatabaseNotFoundException;

class MapsLibraryManager extends MapsBaseManager {

  /**
   * {@inheritdoc}
   */
  protected $data_type = 'library';

  /**
   * {@inheritdoc}
   */
  public function queueBasesBaseQuery(Connection $database, MapsSyncConverterInterface $converter) {
    $query = parent::queueBasesBaseQuery($database, $converter);

    // Check if there is any filters, and manage them.
    foreach ($converter->getFilters() as $filter_name => $filter_value) {
      $query->join('libraries', 'l', "l.id = q.data_id AND $filter_name = '$filter_value'");
    }

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function getBaseFromQueue(Connection $database, MapsSyncConverterInterface $converter, $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array {
    $query = $this->queueBasesBaseQuery($database, $converter)
      ->fields('q', ['profile', 'id', 'data_id'])
      ->orderBy('q.id');

    if ($limit !== NULL) {
      $query->range(0, $limit);
    }

    return $query
      ->execute()
      ->fetchAll(\PDO::FETCH_GROUP|\PDO::FETCH_ASSOC);
  }

  /**
   * {@inheritdoc}
   */
  public function buildBases(MapsSyncConverterInterface $converter, array $object_ids): array {
    try {
      $maps_sync_db = $this->getConnection(TRUE);
      $libraries = [];

      $query = $maps_sync_db
        ->select('libraries', 'l')
        ->fields('l')
        ->condition('l.id', $object_ids, 'IN')
        ->condition('l.profile', $converter->getProfile()->id());

      $result = $query->execute()->fetchAll();

      /** @var MapsLibrary $library */
      foreach ($result as $item) {
        if (!isset($libraries[$item->id])) {
          $library = new MapsLibrary();
          $library
            ->setProfile($item->profile)
            ->setConverter($converter->id())
            ->setId($item->id)
            ->setCode($item->code)
            ->setIdAttribute($item->id_attribute)
            ->setValue([]);

          $libraries[$item->id] = $library;
        }

        $libraries[$item->id]->setValueByLanguage($item->id_language, $item->value);
      }

      return $libraries;
    }
    catch (DatabaseNotFoundException $e) {
      return [];
    }
  }
}
