<?php

namespace Drupal\a12s_maps_sync\Maps;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Entity\MapsSyncProfileInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\DatabaseNotFoundException;
use Drupal\Core\Database\Query\SelectInterface;

/**
 * Class MapsObjectManager
 *
 * // @check Looks very similar to the MapsObjectManager class....
 */
class MapsMediaManager extends MapsBaseManager {

  /**
   * {@inheritdoc}
   */
  protected $data_type = 'media';

  /**
   * {@inheritdoc}
   */
  public function getBaseFromQueue(Connection $database, MapsSyncConverterInterface $converter, $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array {
    $query = $this->queueBasesBaseQuery($database, $converter)
      ->fields('q', ['profile', 'id', 'data_id'])
      ->orderBy('q.id');

    if ($limit !== NULL) {
      $query->range(0, $limit);
    }

    return $query
      ->execute()
      ->fetchAll(\PDO::FETCH_GROUP | \PDO::FETCH_ASSOC);
  }

  /**
   * {@inheritdoc}
   */
  public function buildBases(MapsSyncConverterInterface $converter, array $media_ids): array {
    try {
      $maps_sync_db = $this->getConnection(TRUE);
      $medias = [];

      $result = $maps_sync_db
        ->select('medias', 'm', ['fetch' => MapsMedia::class])
        ->fields('m')
        ->condition('m.id', $media_ids, 'IN')
        ->condition('m.profile', $converter->getProfile()->id())
        ->execute();

      /** @var \Drupal\a12s_maps_sync\Maps\MapsMedia $media */
      foreach ($result as $media) {
        $attributes = [];
        $raw_attributes = $maps_sync_db
          ->select('media_attributes', 'a', ['fetch' => \PDO::FETCH_ASSOC])
          ->fields('a')
          ->condition('a.media_id', $media->getId())
          ->condition('a.profile', $converter->getProfile()->id())
          ->execute();

        foreach ($raw_attributes as $attribute) {
          if (!isset($attributes[$attribute['name']])) {
            $attributes[$attribute['name']] = new MapsAttribute($attribute);
          }
          else {
            $data = @json_decode($attribute['data'], TRUE) ?: [];
            $attributes[$attribute['name']]->setValue($attribute['value'], $attribute['language'], $attribute['delta'], $data);
          }
        }

        $media->setAttributes($attributes);
        $media->setConverter($converter->id());

        $medias[$media->getId()] = $media;
      }

      return $medias;
    } catch (DatabaseNotFoundException $e) {
      return [];
    }
  }

  /**
   * @return array
   */
  public function getFixedMapping(MapsSyncProfileInterface $profile): array {
    return [
      'filename' => 'filename',
      'const:' . $profile->getMediasPath() . '/+filename' => 'uri',
    ];
  }


  /**
   * Modify a query to add a condition on a media attribute.
   *
   * @param \Drupal\Core\Database\Connection $database
   * @param \Drupal\Core\Database\Query\SelectInterface $query
   * @param string $filter_name
   * @param string $filter_value
   */
  public static function addConditionOnMediaAttribute(Connection $database, SelectInterface $query, string $filter_name, string $filter_value) {
    $filter_name = strtoupper(str_replace('attribute_', '', $filter_name));

    if (strpos($filter_value, 'EMPTY') === 0) {
      $notIn = $database->select('media_attributes', 'ma')
        ->fields('ma', ['media_id'])
        ->condition('name', $filter_name)
        ->execute()
        ->fetchCol();

      $query->condition('m.id', $notIn, 'NOT IN');
    }
    else {
      $alias = 'ma_' . $filter_name;
      $query->join('media_attributes', $alias, $alias . '.media_id = m.id');
      $query->condition($alias . '.name', $filter_name);

      if (strpos($filter_value, 'NOT EMPTY') === 0) {
        // Nothing to do, we have set an inner join just before.
      }
      else {
        // Using the id of an attribute library.
        if (strpos($filter_value, 'idattribute_library:') === 0) {
          $filter_value = str_replace('idattribute_library:', '', $filter_value);
          $condition = <<<EOT
%"idattribute_library": "$filter_value"%
EOT;

          $query->condition($alias . '.data', $condition, 'LIKE');
        }
        // Using a simple value.
        else {
          $query->condition($alias . '.value', $filter_value);
        }
      }
    }
  }

  /**
   * Modify a query to add a condition on a media property.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $query
   * @param string $filter_name
   * @param string $filter_value
   */
  public static function addConditionOnMediaProperty(SelectInterface $query, string $filter_name, string $filter_value) {
    $filter_name = str_replace('media_', '', $filter_name);
    $query->condition('m.' . $filter_name, $filter_value);
  }

}
