<?php

namespace Drupal\a12s_maps_sync\Maps;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\DatabaseNotFoundException;

/**
 * Class MapsObjectManager
 */
class MapsObjectManager extends MapsBaseManager {

  /**
   * {@inheritdoc}
   */
  protected $data_type = 'object';

  /**
   * {@inheritdoc}
   */
  public function getBaseFromQueue(Connection $database, MapsSyncConverterInterface $converter, $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array {
    $query = $this->queueBasesBaseQuery($database, $converter)
      ->fields('q', ['profile', 'id', 'data_id'])
      ->orderBy('q.id');

    if ($limit !== NULL) {
      $query->range(0, $limit);
    }

    return $query
      ->execute()
      ->fetchAll(\PDO::FETCH_GROUP|\PDO::FETCH_ASSOC);
  }

  /**
   * {@inheritdoc}
   */
  public function buildBases(MapsSyncConverterInterface $converter, array $object_ids): array {
    try {
      $maps_sync_db = $this->getConnection(TRUE);
      $objects = [];

      $result = $maps_sync_db
        ->select('objects', 'o', ['fetch' => MapsObject::class])
        ->fields('o')
        ->condition('o.id', $object_ids, 'IN')
        ->condition('o.profile', $converter->getProfile()->id())
        ->execute();

      /** @var MapsObject $object */
      foreach ($result as $object) {
        $attributes = [];
        $raw_attributes = $maps_sync_db
          ->select('object_attributes', 'a', ['fetch' => \PDO::FETCH_ASSOC])
          ->fields('a')
          ->condition('a.object_id', $object->getId())
          ->condition('a.profile', $converter->getProfile()->id())
          ->execute();

        foreach ($raw_attributes as $attribute) {
          if (!isset($attributes[$attribute['name']])) {
            $attribute['converter'] = $converter->id();
            $attributes[$attribute['name']] = new MapsAttribute($attribute);
          }
          else {
            $data = @json_decode($attribute['data'], TRUE) ?: [];
            $attributes[$attribute['name']]->setValue($attribute['value'], $attribute['language'], $attribute['delta'], $data);
          }
        }

        $object->setAttributes($attributes);

        // Get the medias.
        $query = $maps_sync_db
          ->select('medias', 'm', ['fetch' => \PDO::FETCH_ASSOC])
          ->fields('m');

        $query->join('object_media', 'oa', 'm.id = oa.media_id');
        $query->condition('oa.object_id', $object->getId());

        $medias = $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
        $object->setMedias($medias);
        $object->setConverter($converter->id());

        $objects[$object->getId()] = $object;
      }

      return $objects;
    }
    catch (DatabaseNotFoundException $e) {
      return [];
    }
  }

}
