<?php

namespace Drupal\a12s_maps_sync\Plugin\MapsSyncHandler;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Event\ConverterItemImportEvent;
use Drupal\a12s_maps_sync\Maps\MapsBaseInterface;
use Drupal\a12s_maps_sync\Plugin\MapsSyncHandlerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Class MediaHandler
 *
 * Handler used for files when we want to create the related media on the fly.
 *
 * @MapsSyncHandler(
 *   id = "media",
 *   label = @Translation("Media")
 * )
 */
class MediaHandler extends MapsSyncHandlerBase {

  /**
   * Get the media types mapping.
   *
   * @return array
   */
  public function getMediaTypesMapping(): array {
    return [
      1 => [
        'type' => 'image',
        'field' => 'field_image',
      ],
      2 => [
        'type' => 'document',
        'field' => 'field_document',
      ],
    ];
  }

  /**
   * Get the media type mapping for a given object.
   *
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @return array
   */
  public function getMediaTypeMapping(MapsBaseInterface $object): array {
    return $this->getMediaTypesMapping()[$object->get('type')];
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL) {
    // Create the media.
    // To find the media type, we are using the media type from MaPS.
    $media_type_mapping = $this->getMediaTypeMapping($object);

    // Check if the entity exists.
    $maps_manager = $converter->getMapsManager();
    $media = $maps_manager->getEntity('media', $media_type_mapping['type'], $object, $converter->getGid(), $language);

    $media->{$media_type_mapping['field']}->target_id = $entity->id();
    $media->set(MapsBaseInterface::GID_FIELD, $entity->get(MapsBaseInterface::GID_FIELD)->value);
    $media->save();

    // We need to re-trigger the dispatch of the item import finished event here.
    $event_dispatcher = \Drupal::service('event_dispatcher');
    $event = new ConverterItemImportEvent($converter, $object, $media, $language);
    $event_dispatcher->dispatch($event, ConverterItemImportEvent::FINISHED);
  }

  /**
   * @inheritDoc
   */
  public function postDelete(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL): void {
    $media_type_mapping = $this->getMediaTypeMapping($object);

    // Check if the entity exists.
    $maps_manager = $converter->getMapsManager();
    $media = $maps_manager->getEntity('media', $media_type_mapping['type'], $object, $converter->getGid(), $language);
    $media->delete();
  }

}
