<?php

namespace Drupal\a12s_maps_sync\Plugin;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverter;
use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Entity\MapsSyncProfileInterface;
use Drupal\a12s_maps_sync\Maps\Exception\MapsEntityDefinitionException;
use Drupal\a12s_maps_sync\Maps\MapsBaseInterface;
use Drupal\a12s_maps_sync\Maps\MapsBaseManager;
use Drupal\a12s_maps_sync\Maps\MapsBaseManagerInterface;
use Drupal\a12s_maps_sync\Maps\MapsMediaManager;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Logger\LoggerChannelTrait;

/**
 * Base class for MaPS Sync handler plugins.
 */
abstract class MapsSyncHandlerBase extends PluginBase implements MapsSyncHandlerInterface {

  use LoggerChannelTrait;

  /**
   * {@inheritDoc}
   */
  public function importData(MapsSyncProfileInterface $profile, array $converters, int $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array {
    $results = [];

    /** @var \Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface $converter */
    foreach ($converters as $converter) {
      $results = $converter->import($limit);
    }
    // @todo remove items from queue.

    $converter->getMapsManager()->closeConnection();
    return $results;
  }

  /**
   * {@inheritDoc}
   */
  public function postDelete(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL): void {
    // Nothing to do...
  }

  /**
   * {@inheritDoc}
   *
   * @param MapsBaseInterface $object
   * @param MapsSyncConverterInterface $target_converter
   * @param LanguageInterface|null $language
   *
   * @return EntityInterface|null
   *
   * @throws MapsEntityDefinitionException
   *   When the converter definition is incomplete.
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function convertItem(MapsBaseInterface $object, MapsSyncConverterInterface $target_converter, LanguageInterface $language = NULL): ?EntityInterface {
    $maps_manager = $target_converter->getMapsManager();

    if (!$entity = $maps_manager->getEntity($target_converter->getConverterEntityType(), $target_converter->getConverterBundle(), $object, $target_converter->getGid(), $language)) {
      return NULL;
    }

    // Manage the entity status.
    $status_mgmt = $target_converter->getStatusManagement() ?? MapsSyncConverterInterface::STATUS_MANAGEMENT_UNPUBLISH;

    if ($target_converter->getMapsType() === 'object') {
      // Special case: only update existing entities, without changing the status.
      if ($status_mgmt === MapsSyncConverterInterface::STATUS_MANAGEMENT_UPDATE_EXISTING_IGNORE_STATUS) {
        $gid_definition = $target_converter->getGid();

        if (empty($gid_definition)) {
          throw new MapsEntityDefinitionException($target_converter->getConverterEntityType(), 'gid');
        }

        // Only update entities, no creation here.
        if ($gid = $target_converter->getMapsManager()
          ->getGid($gid_definition, $object)) {
          $storage = \Drupal::service('entity_type.manager')
            ->getStorage($target_converter->getConverterEntityType());

          if (empty($storage->loadByProperties([MapsBaseInterface::GID_FIELD => $gid]))) {
            return NULL;
          }
        }
      }
      // Ensure that the entity has a "setPublished" method.
      elseif ($status_mgmt === MapsSyncConverterInterface::STATUS_MANAGEMENT_UNPUBLISH && method_exists($entity, 'setPublished') && method_exists($entity, 'setUnpublished')) {
        // Check if the entity has a published status.
        if (in_array((int) $object->get('status'), $target_converter->getPublishedStatuses(), TRUE)) {
          $entity->setPublished();
        }
        else {
          $entity->setUnpublished();
        }
      }
      else {
        if (method_exists($entity, 'setPublished')) {
          $entity->setPublished();
        }

        if (!in_array((int) $object->get('status'), $target_converter->getPublishedStatuses(), TRUE)) {
          $entity->delete();
          $this->postDelete($entity, $object, $target_converter, $language);
          return NULL;
        }
      }
    }
    elseif ($target_converter->getMapsType() === 'media') {
      if ($status_mgmt === MapsSyncConverterInterface::STATUS_MANAGEMENT_DELETE) {
        // Get the status "value".
        if ($object->get($target_converter->getMediaStatusProperty()) !== $target_converter->getMediaStatusPublishedValue()) {
          $entity->delete();
          $this->postDelete($entity, $object, $target_converter, $language);
          return NULL;
        }
      }
    }

    $mapping = $maps_manager->getFixedMapping($target_converter->getProfile());
    $mapping = array_merge($mapping, $target_converter->getMapping());

    if (!empty($mapping)) {
      foreach ($mapping as $source => $targets) {
        // In some cases, we may have multiple Drupal fields mapped to a
        // single MaPS attribute.
        if (!isset($targets['multiple'])) {
          $targets = ['multiple' => [$targets]];
        }

        foreach ($targets['multiple'] as $target) {
          // Manage cases with a format defined (for example: html_full values).
          if (!empty($target['format']) && !empty($target['field_name'])) {
            $format = $target['format'];
            $target = $target['field_name'];

            $entity->get($target)->format = $format;
          }

          if ($source === 'medias') {
            $this->mapMedias($entity, $maps_manager, $object, $target);
          }
          else {
            if (!empty($target['converter'])) {
              $this->mapEntityReferences($entity, $maps_manager, $object, $source, $target);
            }
            // Classic case of simple value.
            else {
              $this->mapValues($entity, $object, $target_converter, $source, $target, $language);
            }
          }
        }
      }
    }

    return $entity;
  }

  /**
   * Map the medias.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseManagerInterface $maps_manager
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @param $target
   */
  public function mapMedias(EntityInterface $entity, MapsBaseManagerInterface $maps_manager, MapsBaseInterface $object, $target) {
    // Medias mapping may define multiple mapping (with different converters).
    if (isset($target['converter'])) {
      $target = [$target];
    }

    foreach ($target as $_target) {
      $values = [];

      // First we load the converter.
      $target_converter = MapsSyncConverter::load($_target['converter']);
      if ($target_converter !== NULL) {
        // Check if the media pass the converter filters.
        // We need to re-retrieve medias for the converter, since some medias
        // may have been deleted, or some medias may not pass the converter
        // conditions anymore.

        $mids = [];
        foreach ($object->getMedias() as $media) {
          $mids[] = $media['id'];
        }

        if (empty($mids)) {
          return;
        }

        $database = $target_converter->getMapsManager()
          ->getConnection(TRUE);

        $query = $database
          ->select('medias', 'm')
          ->fields('m')
          ->condition('m.id', $mids, 'IN');

        // Apply the converter filters manually.
        foreach ($target_converter->getFilters() as $filter_name => $filter_value) {
          if (strpos($filter_name, 'attribute_') === 0) {
            MapsMediaManager::addConditionOnMediaAttribute($database, $query, $filter_name, $filter_value);
          }
          else {
            if (strpos($filter_name, 'media_') === 0) {
              MapsMediaManager::addConditionOnMediaProperty($query, $filter_name, $filter_value);
            }
          }
        }

        $medias = $query
          ->execute()
          ->fetchAll(\PDO::FETCH_GROUP | \PDO::FETCH_ASSOC);

        // Get the medias for the current profile.
        if (!empty($medias)) {
          foreach (array_keys($medias) as $media) {
            $array = [
              'id' => $media,
              'profile' => $target_converter->getProfile()->id(),
              'converter' => $target_converter->id(),
            ];

            // Try to load the target entity.
            if ($gid = $maps_manager->getGidFromArray($target_converter->getGid(), $array)) {
              $storage = \Drupal::service('entity_type.manager')
                ->getStorage('media');
              $entities = $storage->loadByProperties([MapsBaseInterface::GID_FIELD => $gid]);

              if (!empty($entities)) {
                // Set the field value.
                $target_entity = reset($entities);
                $values[] = ['target_id' => $target_entity->id()];
              }
            }
          }
        }
      }

      $entity->{$_target['field_name']} = $values;
    }
  }

  /**
   * Map the entity references.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseManagerInterface $maps_manager
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @param $source
   * @param $target
   */
  public function mapEntityReferences(EntityInterface $entity, MapsBaseManagerInterface $maps_manager, MapsBaseInterface $object, $source, $target) {
    // First we load the converter.
    $target_converter = MapsSyncConverter::load($target['converter']);
    if ($target_converter !== NULL) {
      // We need to find the entity in Drupal.
      $entity_type = !empty($target['entity_type']) ? $target['entity_type'] : $target_converter->getConverterEntityType();

      /** @var \Drupal\Core\Entity\EntityFieldManager $entityFieldManager */
      $entityFieldManager = \Drupal::service('entity_field.manager');
      $fields = $entityFieldManager->getFieldStorageDefinitions($entity->getEntityType()
        ->id());

      // Ensure that we have a field name configured in the mapping.
      if (empty($target['field_name'])) {
        $this->getLogger('a12s_maps_sync')
          ->error('No target field name defined for target converter %converter', ['%converter' => $target_converter->id()]);
        return FALSE;
      }

      if (empty($fields[$target['field_name']]) || $fields[$target['field_name']] === NULL) {
        $this->getLogger('a12s_maps_sync')
          ->error('The field %field_name does not exist.', ['%field_name' => $target['field_name']]);
        return FALSE;
      }

      if ($target['field_name'] === 'parent') {
        $is_multiple = FALSE;
      }
      else {
        $is_multiple = $fields[$target['field_name']]->get('cardinality') !== 1;
      }

      // @todo manage this in a clean way, this is too "hacky".
      $target_gid = $target_converter->getGid();

      // Delete the previous value.
      if (!isset($target['append']) || !$target['append']) {
        $entity->{$target['field_name']} = [];
      }

      for ($i = 0; $i < $object->getCountValues($source); $i++) {
        $array = [];

        foreach ($target_gid as $elem) {
          if (substr($elem, 0, 6) === 'const:') {
            $array[$elem] = explode(':', $elem)[1];
          }
          else {
            switch ($elem) {
              case 'profile':
                $array[$elem] = $target_converter->getProfile()->id();
                break;
              case 'converter':
                $array[$elem] = $target_converter->id();
                break;
              case 'code':
                // We have to load the maps object.
                $targetObjects = $maps_manager->buildBases($target_converter, [$object->get($source, NULL, $i)]);

                /** @var MapsBaseInterface|false $targetObject */
                if ($targetObject = reset($targetObjects)) {
                  $array[$elem] = $targetObject->get($elem);
                }
                else {
                  $this->getLogger('a12s_maps_sync')
                    ->error('Cannot retrieve target object using GID based on code, for the MaPS object %object_id (source: %source, converter: %converter).', [
                      '%converter' => $target_converter->id(),
                      '%object_id' => $object->getId(),
                      '%source' => $source,
                    ]);
                }

                break;
              default:
                $array[$elem] = $object->get($source, NULL, $i);
            }
          }
        }

        // Try to load the target entity.
        if ($gid = $maps_manager->getGidFromArray($target_converter->getGid(), $array)) {
          $storage = \Drupal::service('entity_type.manager')
            ->getStorage($entity_type);
          $entities = $storage->loadByProperties([MapsBaseInterface::GID_FIELD => $gid]);

          if (!empty($entities)) {
            // Set the field value.
            $target_entity = reset($entities);
            if ($is_multiple) {
              $entity->{$target['field_name']}[] = $target_entity->id();
            }
            else {
              $entity->{$target['field_name']} = $target_entity->id();
            }
          }
          else {
            // Ultra special case for parents.
            if ($target['field_name'] === 'parent') {
              // Set to root.
              $entity->get($target['field_name'])->target_id = 0;
            }
          }
        }
      }
    }
  }

  /**
   * Map the "classic" values.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @param \Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface $target_converter
   * @param $source
   * @param $target
   * @param $language
   */
  public function mapValues(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $target_converter, $source, $target, $language) {
    $language_id = $language !== NULL && $entity->isTranslatable()
      ? \Drupal::config('a12s_maps_sync.languages_mapping')
        ->get($language->getId())
      : $target_converter->getProfile()->getDefaultMapsLanguage();
    //      : \Drupal::config('a12s_maps_sync.languages_mapping')->get($target_converter->getProfile()->getDefaultMapsLanguage());

    // @todo Use tokens.
    // @todo Use hasField? Leave Exception be thrown?
    $parts = explode('+', $source);
    foreach ($parts as &$part) {
      if (strpos($part, 'const:') === 0) {
        $part = str_replace('const:', '', $part);
      }
      else {
        $part = $object->get($part, $language_id);
      }
    }
    $value = implode('', $parts);

    // Here, we have to manage a special case for empty values.
    // If the field is translatable, and we are not currently on the
    // default language, we must not reset the value.
    if ($entity->isTranslatable() && empty($value) && $language !== NULL && $language->getId() !== \Drupal::languageManager()
        ->getDefaultLanguage()
        ->getId()) {
      return;
    }

    $entity->get($target)->value = $value;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL) {
    // If the entity is translatable and the label empty, we try to use
    // the label of the default language.
    if ($entity->isTranslatable() && empty($entity->label())) {
      $default = $entity->getTranslation(\Drupal::languageManager()
        ->getDefaultLanguage()
        ->getId());
      $label = $default->label();

      $entity->set($entity->getEntityType()->getKey('label'), $label);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL) {
  }

}
