<?php

namespace Drupal\a12s_maps_sync\Plugin;

use Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface;
use Drupal\a12s_maps_sync\Entity\MapsSyncProfileInterface;
use Drupal\a12s_maps_sync\Maps\MapsBaseInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;

/**
 * Interface MapsSyncHandlerInterface.
 */
interface MapsSyncHandlerInterface {

  /**
   * The default number of items to be fetched for each import process.
   */
  public const QUERY_LIMIT = 20;

  /**
   * Import data from MaPS System.
   *
   * @param MapsSyncProfileInterface $profile
   *   The profile.
   * @param array $converters
   *   The list of converter settings that may apply.
   * @param int $limit
   *   The maximum number of objects to retrieve.
   *
   * @return array
   *   The list of processed items. Each item may contain the following keys:
   *   - object: The MaPS Object instance.
   *   - profile: The profile name.
   *   - converter: The converter that was used for the convert process.
   *   - entity_type: The target entity type.
   *   - entity_id: The converted entity ID. Default to NULL.
   *   - success: Whether the conversion is successful or not. On success, the
   *   value is either SAVED_NEW or SAVED_UPDATED, depending on the operation
   *   performed on the Drupal entity. Default to FALSE.
   *   - gid: The calculated GID. Default to NULL.
   *
   * Contributed modules may add some keys to this array.
   */
  public function importData(MapsSyncProfileInterface $profile, array $converters, int $limit = MapsSyncHandlerInterface::QUERY_LIMIT): array;

  /**
   * Convert given object to Drupal content.
   *
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   *   The MaPS System object
   * @param MapsSyncConverterInterface $target_converter
   *   The converter settings.
   *
   * @param \Drupal\Core\Language\LanguageInterface|null $language
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The converted entity, not still saved.
   */
  public function convertItem(MapsBaseInterface $object, MapsSyncConverterInterface $target_converter, LanguageInterface $language): ?EntityInterface;

  /**
   * Allow additional process before saving the Entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @param \Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface $converter
   * @param \Drupal\Core\Language\LanguageInterface $language
   * @return mixed
   */
  public function preSave(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL);

  /**
   * Allow additional process after saving the Entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @param \Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface $converter
   * @param \Drupal\Core\Language\LanguageInterface $language
   * @return mixed
   */
  public function postSave(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL);

  /**
   * Allow additional process after deleting the Entity.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   * @param \Drupal\a12s_maps_sync\Maps\MapsBaseInterface $object
   * @param \Drupal\a12s_maps_sync\Entity\MapsSyncConverterInterface $converter
   * @param \Drupal\Core\Language\LanguageInterface|null $language
   */
  public function postDelete(EntityInterface $entity, MapsBaseInterface $object, MapsSyncConverterInterface $converter, LanguageInterface $language = NULL): void;

}
