(function (Drupal, once) {
  Drupal.behaviors.trackingCodeSuggestions = {
    attach: function (context, settings) {
      console.log('📦 trackingcode_suggestions geladen');

      /**
       * Retrieve the content from a wrapper.
       * First, we try to detect whether a link field is present (e.g., via
       * an entity reference that is displayed as a link). If so, we retrieve
       * the value of that field (or fall back to the textContent). Otherwise,
       * we check for CKEditor fields, textareas, input fields, or otherwise the
       * entire wrapper.
       */
      function getContentFromWrapper(wrapper) {
        // If a link field is present, use the value of the text field.
        var linkField = wrapper.querySelector('.field--type-link');
        if (linkField) {
          var input = linkField.querySelector('input[type="text"]');
          if (input && input.value.trim() !== '') {
            return input.value.trim();
          }
          // Fallback: use the text content of the link field.
          return linkField.textContent.trim();
        }
        // If a CKEditor 5 field is present, use the innerHTML of the editable element.
        var editableEl = wrapper.querySelector('.ck-editor__editable');
        if (editableEl) {
          return editableEl.innerHTML;
        }
        // If a textarea or input is present, use its value.
        var field = wrapper.querySelector('textarea, input');
        if (field) {
          return field.value;
        }
        // As a final fallback, use the entire innerHTML of the wrapper.
        return wrapper.innerHTML;
      }

      /**
       * Parse an HTML string and return an array with all valid hrefs.
       * If no <a> tags are found but the string itself looks like a URL,
       * that string is added as a link.
       */
      function getLinksFromHTML(html) {
        var tempDiv = document.createElement('div');
        tempDiv.innerHTML = html;
        var links = tempDiv.querySelectorAll('a[href]');
        var hrefs = Array.from(links)
          .map(function(link) {
            return link.href;
          })
          .filter(function(href) {
            return href && !href.startsWith('javascript:');
          });
        // If there is no <a> tag but the text itself looks like a URL
        if (hrefs.length === 0) {
          var trimmed = html.trim();
          if (/^(https?:\/\/|\/)/i.test(trimmed)) {
            hrefs.push(trimmed);
          }
        }
        return hrefs;
      }

      /**
       * Generate the tracking code suggestions based on the links in the content
       * and display them in the tracking code field
       */
      function showSuggestions() {
        console.log('🔍 Get new suggestions');

        // Retrieve unique ID from ab-test wrapper.
        var uuidField = context.querySelector('#edit-field-layout-0-subform-field-unique-id-0-value');
        var uuid = uuidField ? uuidField.value.trim() : '';

        // Define the variants with their corresponding selectors.
        var variants = [
          {
            name: 'A',
            wrapperSelector: '[data-drupal-selector*="field-variant-a"]',
            targetFieldSelector: '[data-drupal-selector*="field-tracking-code-a"]'
          },
          {
            name: 'B',
            wrapperSelector: '[data-drupal-selector*="field-variant-b"]',
            targetFieldSelector: '[data-drupal-selector*="field-tracking-code-b"]'
          }
        ];

        variants.forEach(function(variant) {
          var wrapper = context.querySelector(variant.wrapperSelector);
          if (!wrapper) {
            console.warn('No wrapper found for variant ' + variant.name);
            return;
          }

          // Retrieve the content (whether it's a CKEditor, link field, textarea, or input).
          var content = getContentFromWrapper(wrapper);
          if (!content) {
            console.warn('No wrapper found for variant ' + variant.name);
            return;
          }

          // Find all links in the content.
          var links = getLinksFromHTML(content);
          console.log('🔗 Links found in variant ' + variant.name + ':', links);

          // Build a tracking code suggestion for each link.
          var suggestions = links.map(function(href) {
            var code = "_paq.push(['trackEvent', 'A/B Test', 'Click', '" + uuid + " - Variant " + variant.name + "']);";
            return { code: code, href: href };
          });

          // Find the container and the input field for the tracking code.
          var targetContainer = context.querySelector(variant.targetFieldSelector + ' .form-item__description');
          var targetInput = context.querySelector(variant.targetFieldSelector + ' input, ' + variant.targetFieldSelector + ' textarea');

          if (targetContainer && suggestions.length > 0) {
            var suggestionsHTML = suggestions.map(function(s) {
              return '<div style="margin-top: 0.4em;"><code>' + s.code + '</code> ' +
                '<button class="add-suggestion" data-code="' + encodeURIComponent(s.code) + '" style="margin-left: 0.5em;">📋 Kopieer</button>' +
                '</div>';
            }).join('');
            targetContainer.innerHTML = '<strong>📌 Suggesties:</strong>' + suggestionsHTML;

            // Add event listeners: on click, the code is copied to the clipboard.
            targetContainer.querySelectorAll('.add-suggestion').forEach(function(btn) {
              btn.addEventListener('click', function(e) {
                e.preventDefault();
                var code = decodeURIComponent(e.target.dataset.code);
                navigator.clipboard.writeText(code).then(function() {
                  console.log('Tracking code successfully copied to the clipboard.');
                }).catch(function(err) {
                  console.error('Copy action failed:', err);
                });
              });
            });
          }
        });
      }

      /**
       * If there are changes in CKEditor fields or other input fields,
       * update the suggestions.
       */
      function attachObservers() {
        // For CKEditor fields.
        var editables = context.querySelectorAll('.ck-editor__editable');
        editables.forEach(function(editable) {
          once('ckeditor-mutation-observer', editable).forEach(function(el) {
            var observer = new MutationObserver(function() {
              setTimeout(showSuggestions, 300);
            });
            observer.observe(editable, { childList: true, subtree: true, characterData: true });
          });
        });
        // For other input fields: textarea, input, select.
        once('suggestion-listeners', 'textarea, input, select', context).forEach(function(el) {
          el.addEventListener('blur', function() {
            setTimeout(showSuggestions, 300);
          });
        });
      }

      // Run initially.
      showSuggestions();
      attachObservers();
    }
  };
})(Drupal, once);
