(function (Drupal, once) {
  Drupal.behaviors.trackingCodeSuggestions = {
    attach: function (context, settings) {
      console.log('📦 trackingcode_suggestions loaded');

      // Determine suggestion type from Drupal settings: 'matomo' or 'google_analytics'.
      var provider =
        settings.abParagraphsSettings &&
        settings.abParagraphsSettings.analyticsProvider
          ? settings.abParagraphsSettings.analyticsProvider
          : 'matomo';

      // Debug it:
      console.log('⚙️ analyticsProvider =', provider);

      /**
       * Retrieve the content from a wrapper.
       */
      function getContentFromWrapper(wrapper) {
        // Link field (entity reference) first.
        var linkField = wrapper.querySelector('.field--type-link');
        if (linkField) {
          var input = linkField.querySelector('input[type="text"]');
          if (input) {
            // Use data-original-value if available (e.g. "/node/123")
            var originalValue = input.getAttribute('data-resolved-path');
            if (originalValue && originalValue.trim() !== '') {
              return originalValue.trim();
            }
            // Fallback to regular input value
            if (input.value.trim() !== '') {
              return input.value.trim();
            }
          }
          return linkField.textContent.trim();
        }

        // CKEditor 5
        var editableEl = wrapper.querySelector('.ck-editor__editable');
        if (editableEl) {
          return editableEl.innerHTML;
        }

        // Textarea / input
        var field = wrapper.querySelector('textarea, input');
        if (field) {
          return field.value;
        }

        // Fallback: full HTML
        return wrapper.innerHTML;
      }

      /**
       * Parse an HTML string and return all hrefs (absolute or relative).
       */
      function getLinksFromHTML(html) {
        var tempDiv = document.createElement('div');
        tempDiv.innerHTML = html;
        var anchors = tempDiv.querySelectorAll('a[href]');
        var hrefs = Array.from(anchors)
          .map(function(a) { return a.getAttribute('href'); })
          .filter(function(h) { return h && !h.startsWith('javascript:'); });
        // If no anchors but the whole string looks like a URL or path:
        if (hrefs.length === 0) {
          var t = html.trim();
          if (/^(https?:\/\/|\/)/i.test(t)) {
            hrefs.push(t);
          }
        }
        return hrefs;
      }

      /**
       * Build & render tracking‑code suggestions.
       */
      function showSuggestions() {
        console.log('🔍 Generating new suggestions…');

        // Grab the “Unique ID” field by its data‑drupal‑selector.
        var uuidField = context.querySelector('[data-drupal-selector="edit-field-unique-id-0-value"]');
        var uuid = uuidField ? uuidField.value.trim() : '';

        var variants = [
          {
            name: 'A',
            wrapperSelector: '[data-drupal-selector*="field-variant-a"]',
            targetFieldSelector: '[data-drupal-selector*="field-tracking-code-a"]'
          },
          {
            name: 'B',
            wrapperSelector: '[data-drupal-selector*="field-variant-b"]',
            targetFieldSelector: '[data-drupal-selector*="field-tracking-code-b"]'
          }
        ];

        variants.forEach(function(variant) {
          var wrapper = context.querySelector(variant.wrapperSelector);
          if (!wrapper) {
            console.warn('No wrapper for variant', variant.name);
            return;
          }

          var content = getContentFromWrapper(wrapper);
          if (!content) {
            console.warn('Empty content for variant', variant.name);
            return;
          }

          var links = getLinksFromHTML(content);
          console.log('🔗 Links for variant', variant.name, ':', links);

          // Build suggestions for each link, using the selected analytics library.
          var suggestions = links.map(function(href) {
            var code;
            if (provider === 'google_analytics') {
              code = "gtag('event', 'click', {" +
                "'event_category':'A/B Test'," +
                "'event_label':'" + href + " - " + uuid + " - Variant " + variant.name + "'" +
                "});";
            }
            else {
              code = "_paq.push(['trackEvent', 'A/B Test', 'Click', '" +
                href + " - " + uuid + " - Variant " + variant.name +
                "']);";
            }
            return { code: code, href: href };
          });

          var targetContainer = context.querySelector(
            variant.targetFieldSelector + ' .form-item__description'
          );
          // Enforce max-width = 100%
          if (targetContainer) {
            targetContainer.style.maxWidth = '100%';
          }
          if (!targetContainer || suggestions.length === 0) {
            return;
          }

          // Render header + refresh button + each suggestion
          var header = '<strong>📌 Suggestions:</strong> ' +
            '<button type="button" class="refresh-suggestions" ' +
            'style="margin-left:0.5em;">🔄 Refresh</button>';
          var items = suggestions.map(function(s) {
            return '<div style="margin-top:0.4em;">' +
              '<code>' + s.code + '</code> ' +
              '<button type="button" class="add-suggestion" ' +
              'data-code="' + encodeURIComponent(s.code) + '" ' +
              'style="margin-left:0.5em;">📋 Copy</button>' +
              '</div>';
          }).join('');

          targetContainer.innerHTML = header + items;

          // Copy‑to‑clipboard handlers
          targetContainer.querySelectorAll('.add-suggestion').forEach(function(btn) {
            btn.addEventListener('click', function(e) {
              e.preventDefault();
              var code = decodeURIComponent(btn.dataset.code);
              navigator.clipboard.writeText(code)
                .then(function() { console.log('✅ Code copied'); })
                .catch(function(err) { console.error('❌ Copy failed', err); });
            });
          });

          // Refresh handler
          var refreshBtn = targetContainer.querySelector('.refresh-suggestions');
          if (refreshBtn) {
            refreshBtn.addEventListener('click', function(e) {
              e.preventDefault();
              showSuggestions();
            });
          }
        });
      }

      /**
       * Watch for changes (CKEditor mutations + blur on inputs) to regen suggestions.
       */
      function attachObservers() {
        // CKEditor 5
        context.querySelectorAll('.ck-editor__editable').forEach(function(editable) {
          once('ckeditor-mutation-observer', editable).forEach(function(el) {
            var mo = new MutationObserver(function() {
              setTimeout(showSuggestions, 300);
            });
            mo.observe(el, { childList: true, subtree: true, characterData: true });
          });
        });
        // Other fields
        once('suggestion-listeners', 'textarea, input, select', context).forEach(function(el) {
          el.addEventListener('blur', function() {
            setTimeout(showSuggestions, 300);
          });
        });
      }

      // kick things off
      showSuggestions();
      attachObservers();
    }
  };
})(Drupal, once);
