(function ($, Drupal, drupalSettings, once) {
  var DataLayer = window.dataLayer || [];
  var phone_hash = drupalSettings.user_phone ?? '';

  Drupal.behaviors.map = {
    attach: function (context, settings) {

      function isPostcode(input) {
        const postcode = input.trim().toUpperCase();

        const patterns = [
          // United Kingdom (complex format)
          /^([A-Z]{1,2}\d[A-Z\d]? ?\d[A-Z]{2})$/,

          // United States (ZIP / ZIP+4)
          /^\d{5}(-\d{4})?$/,

          // Canada
          /^[A-Z]\d[A-Z][ ]?\d[A-Z]\d$/,

          // Germany, France, Netherlands, Austria, Spain, Finland, Belgium (mostly 4-5 digit codes)
          /^\d{4,5}$/,

          // Italy (5 digit)
          /^\d{5}$/,

          // Switzerland (4 digit)
          /^\d{4}$/,

          // Norway (4 digit)
          /^\d{4}$/,

          // Sweden (5 digit, often with space)
          /^\d{3}\s?\d{2}$/,

          // Denmark (4 digit)
          /^\d{4}$/,

          // Netherlands (4 digits + 2 letters, optional space)
          /^\d{4}\s?[A-Z]{2}$/
        ];

        return patterns.some((regex) => regex.test(postcode));
      }

      async function getCoordsByPostcode(postcode) {
        const encoded = encodeURIComponent(postcode);
        const url = `https://nominatim.openstreetmap.org/search?format=json&countrycodes=gb&q=${encoded}`;

        const response = await fetch(url, {
          headers: {
            'User-Agent': 'abinbev_gmap/1.0'
          }
        });

        const data = await response.json();
        if (data.length > 0) {
          return {
            lat: parseFloat(data[0].lat),
            lon: parseFloat(data[0].lon),
          };
        }

        return null;
      }

      // 1. Получаем координаты по UK postcode
      async function getCoordsByUKPostcode(postcode) {
        const cleanedPostcode = postcode.replace(/\s/g, '');
        const url = `https://api.postcodes.io/postcodes/${cleanedPostcode}`;

        const response = await fetch(url);
        const data = await response.json();

        if (data.status === 200 && data.result) {
          return {
            lat: data.result.latitude,
            lng: data.result.longitude
          };
        } else {
          console.warn("Postcode not found");
          return null;
        }
      }

      // 2. Считаем расстояние между двумя координатами (в км)
      function getDistance(lat1, lon1, lat2, lon2) {
        const R = 6371; // Радиус Земли в км
        const dLat = (lat2 - lat1) * Math.PI / 180;
        const dLon = (lon2 - lon1) * Math.PI / 180;
        const a =
          Math.sin(dLat / 2) ** 2 +
          Math.cos(lat1 * Math.PI / 180) *
          Math.cos(lat2 * Math.PI / 180) *
          Math.sin(dLon / 2) ** 2;
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c;
      }

      // 3. Основная функция: поиск точек в радиусе
      async function findNearbyPointsFromPostcode(postcode, allPoints, radiusKm = 10, center = {}) {
        let coords = {};
        if (postcode) {
          coords = await getCoordsByUKPostcode(postcode);
          if (!coords) {
            console.log("Postcode not found");
            return [];
          }
        }

        if (center) {
          coords = center
        }

        const { lat, lon } = {lat: coords.lat, lon: coords.lng};

        return allPoints
          .map(point => ({
            ...point,
            distance: getDistance(lat, lon, point.lat, point.lon)
          }))
          .filter(point => point.distance <= radiusKm)
          .sort((a, b) => a.distance - b.distance);
      }

      // Initialize and add the map
      const maps = once('initMap', '.map', context);
      google.maps.visualRefresh = true;
      maps.forEach(function () {
        var mapWrapper = $(this);
        var map;
        var markerCluster;
        var markers = [];
        var mapSettings = {};
        let locations = drupalSettings.locations !== undefined ? drupalSettings.locations : [];
        let icons = drupalSettings.icons ?? [];
        let map_id = drupalSettings.map_id ?? '';
        let use_clusters = drupalSettings.use_clusters ?? [];
        let show_categories = drupalSettings.show_categories ?? [];
        let show_list_counter = drupalSettings.show_list_counter ?? [];
        let show_directions = drupalSettings.show_directions ?? [];
        let show_share = drupalSettings.show_share ?? [];
        let use_numbers = drupalSettings.use_numbers ?? [];
        let use_datalayer = drupalSettings.use_datalayer ?? [];
        let use_select_poc = drupalSettings.use_select_poc ?? [];
        let use_select_poc_link = drupalSettings.use_select_poc_link ?? [];
        let use_select_poc_link_text = drupalSettings.use_select_poc_link_text ?? [];
        let filtered_locations = [];
        mapSettings.default_zoom = 7;
        // var mapSettings = drupalSettings.default_map_settings;

        var infoWindow = {};
        var bounds = {}

        setTimeout(function () {
          initialize();
        }, 100);

        function addYourLocationButton(map, locations) {

          if (!navigator.geolocation) {
            console.warn("Geolocation is not supported by this browser.");
            return;
          }

          const controlDiv = document.createElement("div");
          controlDiv.style.margin = "10px";

          const controlUI = document.createElement("button");
          controlUI.classList.add("your-location-btn");
          controlUI.title = "Click to recenter the map";
          //controlUI.innerText = "📍";
          controlUI.innerHTML = `
  <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">
    <circle cx="12" cy="12" r="3" fill="#AECBFA"/>
    <circle cx="12" cy="12" r="9" stroke="#AECBFA" stroke-width="2"/>
    <line x1="12" y1="2" x2="12" y2="5" stroke="#AECBFA" stroke-width="2"/>
    <line x1="12" y1="19" x2="12" y2="22" stroke="#AECBFA" stroke-width="2"/>
    <line x1="2" y1="12" x2="5" y2="12" stroke="#AECBFA" stroke-width="2"/>
    <line x1="19" y1="12" x2="22" y2="12" stroke="#AECBFA" stroke-width="2"/>
  </svg>
`;
          controlDiv.appendChild(controlUI);

          controlUI.addEventListener("click", function () {

            navigator.geolocation.getCurrentPosition(
              (position) => {
                const pos = {
                  lat: +position.coords.latitude,
                  lng: +position.coords.longitude,
                  // lat: 53.4395223,
                  // lng: -2.9587163
                };


                const bounds = map.getBounds();
                let bound_pos = bounds.getCenter();

                const defaultCenter = (bound_pos.lat() && bound_pos.lng())
                  ? { lat: +bound_pos.lat(), lng: +bound_pos.lng() }
                  : map.getCenter(); // Fallback to current map center

                // Make sure bounds is valid
                if (bounds && !bounds.contains(pos)) {

                  setTimeout(function() {
                    map.panTo(defaultCenter);
                    map.setZoom(6);
                  }, 100);

                } else {
                  map.setCenter(pos);
                  map.setZoom(14);
                }

                findNearbyPointsFromPostcode('', locations, 10, pos).then((locations) => {
                  clearMarkers();
                  renderLocationsOnMap(locations);
                  renderListItem(locations);
                  infoWindow.close();
                });
              },
              () => {
                alert("Geolocation permission denied or unavailable.");
              }
            );
          });

          map.controls[google.maps.ControlPosition.RIGHT_BOTTOM].push(controlDiv);
        }
        function initialize(lat, lon, defaultZoom = '') {
          infoWindow = new google.maps.InfoWindow({
            content: '',
          });

          bounds = new google.maps.LatLngBounds();

          let styles = [
            {
              "featureType": "administrative",
              "elementType": "all",
              "stylers": [
                {
                  "visibility": "simplified"
                }
              ]
            },
            {
              "featureType": "landscape",
              "elementType": "all",
              "stylers": [
                {
                  "visibility": "on"
                }
              ]
            },
            {
              "featureType": "poi",
              "elementType": "all",
              "stylers": [
                {
                  "visibility": "off"
                }
              ]
            },
            {
              "featureType": "transit",
              "elementType": "all",
              "stylers": [
                {
                  "visibility": "off"
                }
              ]
            }
          ];
          let default_lat = lat ? lat : 53.3229432;
          let default_lon = lon ? lon : -7.7483999;
          let initZoom = defaultZoom !== '' ? +defaultZoom : +mapSettings.default_zoom;


          map = new google.maps.Map(
            document.getElementById('map'), {
              center: new google.maps.LatLng(default_lat, default_lon),
              zoom: map !== undefined ? map.getZoom() : initZoom,
              mapTypeId: google.maps.MapTypeId.ROADMAP,
              mapId: map_id,
              //mapId: 'c4c116ea09005722',
              streetViewControl: false,
              gestureHandling: "greedy",
              fullscreenControl: true,
              mapTypeControl: false,
              styles: styles
            });

          addYourLocationButton(map, locations);
          requestPocs();

          map.addListener("click", function (event) {
            infoWindow.close();
            $('#map-info-panel .close').click();

            if (use_datalayer) {
              DataLayer.push({
                'event': 'GAEvent',
                'event_category': 'Map',
                'event_action': 'CloseInfoWindow',
                'event_label': "Close", // example: Know More
                'interaction': 'true',
                'component_name': 'Google maps close infowindow', // example:knowmore_button
                'element_text': 'Google maps close infowindow', // example: Know More
                'user_id_phone_hash': phone_hash
              });
            }
          });
        }

        // Make Ajax call to receive list of pocs.
        function requestPocs() {
          let pocId = 'active';
          if (locations) {
            let activePromise = new Promise((resolve, reject) => {
              renderLocationsOnMap(locations, pocId);
              renderListItem(locations);
              if (drupalSettings.use_filters != undefined && drupalSettings.use_filters) {
                let categories = renderCategories();
              }
            })
          }
        }

        /**
         * Render List Item.
         * @param locations
         * @returns {string}
         */
        function renderListItem(locations) {
          let element = '';
          $('.place-list-wrapper').html('');
          let number = '';
          for (let i = 0; i < locations.length; i++) {
            let location = locations[i];

            let info = JSON.parse(location.full_info);
            let category = '';
            if (show_categories) {
              let category_class = location.category.toLowerCase().split(' ').join('-');
              category = ' | <span class="' + category_class + '">' + location.category + '</span>';
            }
            let buttons_start = '';
            let buttons_end = '';
            if (show_directions || show_share) {
              buttons_start = '<div class="buttons">';
              buttons_end = '</div>';
            }
            let direction_button = '';
            if (show_directions) {
              direction_button = '<a class="direction-button" href="https://www.google.com/maps/dir/Current+Location/' + location.lat + ',' + location.lon + '" target="_blank">' + Drupal.t('Get Directions') + '</a>';
            }

            let share_button = '';
            if (show_share) {
              share_button = ' <a  href="#" class="share-offer-link" data-bs-toggle="modal" data-bs-target="#shareModal" data-place-title="' + location.title + '">' +
                Drupal.t('Share') +
                '</a>';
            }

            let poc_link = '';
            if (use_select_poc) {
              poc_link =  ' <a  href="'+use_select_poc_link+'?pocid='+location.id+'" class="select-poc-link" >' +
                Drupal.t(use_select_poc_link_text) +
                '</a>';
            }

            if (use_numbers) {
              number = (i + 1) + '. ';
            }
            element += '<div class="place-wrapper" data-place-title="' + location.title + '" data-place-id="' + location.place_id + '">' +
              '<h2>' + number + location.title + category + '</h2>' +
              '<p>' + info.formatted_address + '</p>' +
              buttons_start +
              direction_button +
              share_button +
              poc_link +
              buttons_end +
              '</div>' +
              '</div>';

          }

          if (!element.length) {
            element = '<h3>' + Drupal.t("Place has not been found.") + '</h3>';
          }

          if (drupalSettings.no_api_key === true) {
            element = '<h3>' + Drupal.t("No Google API key provided. Please contact site admin to provide it.") + '</h3>';
          }
          $('.place-list-wrapper').append(element);

          if (show_list_counter) {
            $('.total-count-placeholder').html('<div class="total-count">' + Drupal.formatPlural(locations.length, '<span>1</span> bar found', '<span> @count </span> bars found') + '</div>');
          }

          return element;
        }

        // Render Info template.
        function renderInfo(place) {

          let full_info =  JSON.parse(place.full_info);
          let info_content = '';
          info_content = '<h2>' + place.title + '</h2>';
          info_content += '<div class="address"><span class="icon"></span>' + full_info.formatted_address + '</div><br />';
          if (show_directions || show_share) {
            info_content += '<div class="buttons">';
          }
          if (show_directions) {
            info_content += '<div>' + '<a class="direction-button" href="https://www.google.com/maps/dir/Current+Location/' + place.lat + ',' + place.lon + '" target="_blank">' + Drupal.t('Get Directions') + '</a></div>';
          }
          if (show_share) {
            info_content +=
              '<a href="#" class="share-offer-link" data-bs-toggle="modal" data-bs-target="#shareModal" data-place-title="' + place.title + '">' +
              'Share' +
              '</a>';
          }
          if (use_select_poc) {
            info_content +=
              '<a  href="'+use_select_poc_link+'?pocid='+place.id+'" class="select-poc-link" >' +
              Drupal.t(use_select_poc_link_text) +
              '</a>';
          }
          if (show_directions || show_share) {
            info_content += '</div>';
          }
          return info_content;
        }


        // Render Map. Put all points.
        function renderLocationsOnMap(locations, allLocations = [], use_bound = true, use_clusters = true) {
          let places = [];
          bounds = new google.maps.LatLngBounds();
          if (locations.length) {
            for (let i = 0; i < locations.length; i++) {
              places.push(renderMarker(locations[i]));
              var geoCode = new google.maps.LatLng(locations[i]['lat'], locations[i]['lon']);
              if (use_bound) {
                bounds.extend(geoCode);
              }
              openSharedPlace(locations[i]);
            }
          }
          else {
            for (let i = 0; i < allLocations.length; i++) {
              var geoCode = new google.maps.LatLng(allLocations[i]['lat'], allLocations[i]['lon']);
              if (use_bound) {
                bounds.extend(geoCode);
              }
              openSharedPlace(allLocations[i]);
            }
          }
          map.fitBounds(bounds);


          // If use clusters option is enabled.
          if (use_clusters) {
            if (markerCluster) {
              markerCluster.setIgnoreHidden(true);
              markerCluster.repaint();
            }

            markerCluster = new MarkerClusterer(map, places, {
              imagePath: 'https://cdn.rawgit.com/googlemaps/js-marker-clusterer/gh-pages/images/m',
              ignoreHidden: true,
              maxZoom: 12,
              zoomOnClick: true,
            });
          }
        }

        // Make markers invisible.
        function clearMarkers() {
          for (let i = 0; i < markers.length; i++) {
            //marker.setVisible(showMarker);
            markers[i].marker.setVisible(false);
          }
          //scrollToMap();
        }

        /**
         * Render Map. Pure Google APi methods.
         * @param place
         * @param poc_type
         */
        let showMarker = '';

        function renderMarker(place, poc_type = 'active') {
          let icon_path = '';

          if (drupalSettings.default_icon !== undefined && drupalSettings.default_icon.length) {
            icon_path = drupalSettings.default_icon;
          }

          if (drupalSettings.use_categories != undefined && drupalSettings.use_categories) {
            let categories = place.category;
            categories = categories.split(',');
            categories = categories.map(function (item) {
              return item.toLowerCase();
            })

            if (categories.length && drupalSettings.icons[categories[0]] !== undefined) {
              icon_path = drupalSettings.icons[categories[0]].file;
            }
          }

          let lat = +place.lat;
          let lon = +place.lon;
          let zoom = map.getZoom()
          var marker;

          if (icon_path !== "") {
            marker = new google.maps.Marker({
              map: map,
              position: new google.maps.LatLng({lat: lat, lng: lon}),
              title: place.title,
              icon: {
                url: icon_path ?? "",
                scaledSize: new google.maps.Size(30, 36), // scaled size
              },
            });
          } else {
            marker = new google.maps.Marker({
              map: map,
              position: new google.maps.LatLng({lat: lat, lng: lon}),
              title: place.title,
            });
          }
          //showMarker = (poc_type === 'active') || (zoom >= +mapSettings.na_zoom);
          showMarker = true;
          marker.setVisible(showMarker);
          bounds.extend(marker.position);

          markers.push({'marker': marker, 'poc_type': poc_type});

          google.maps.event.addListener(marker, 'click', function () {
            //let user_name = drupalSettings.user_name;
            infoWindow.setContent(renderInfo(place));
            infoWindow.open(map, marker);
            if (use_datalayer) {
              DataLayer.push({
                'event': 'GAEvent',
                'event_category': 'Map',
                'event_action': 'OpenInfoWindow',
                'event_label': "Open",
                'interaction': 'true',
                'component_name': 'open_active_poc_info_window',
                'element_text': 'Open Active POC Info Window',
                'product': place.title, // product description
                'sku': place.poc_id, // product sku
                'user_id_phone_hash': phone_hash
              });
            }


          });
          return marker;
        }

        $('.clear-search').on('click', function () {
          let locations = drupalSettings.locations !== undefined ? drupalSettings.locations : [];
          clearMarkers();
          renderLocationsOnMap(locations);
          renderListItem(locations);
          infoWindow.close();
          $('.place-search-form input').val('');
        })

        $('.search_locations').on('click', function () {
          $('.place-search-form').submit();
        })


        function searchZipForZipSearch(search, locations) {
          locations.filter(function (location) {
            if (search.length && search !== "") {
              let info = JSON.parse(location.full_info)
              let addr = info.formatted_address.toLowerCase().includes(search);

              if (addr) {
                let pos = {lat: +location.lat, lng: +location.lon};
                map.setCenter(pos);
                map.setZoom(15);

              }
            }
          });
        }
        /**
         * Submit serach form.
         */
        $('.place-search-form').on('submit', function (e) {
          e.preventDefault();
          filtered_locations = [];
          let search = $(this).find('input').val().toLowerCase();
          let locations = drupalSettings.locations !== undefined ? drupalSettings.locations : [];
          //let search_values = search.split(' ');
          let search_values = search.trim();
          let isZip = isPostcode(search_values);

          if (isZip) {
            findNearbyPointsFromPostcode(search_values, locations, 10, '').then(locations => {
              clearMarkers();
              renderLocationsOnMap(locations, [],false, false);
              renderListItem(locations);
              searchZipForZipSearch(search_values, locations);
              infoWindow.close();
            });
          }
          else {

            let search_result = locations.filter(function (location) {
              // for (let i = 0; i < search_values.length; i++) {
              //let search = search_values[i];
              let search = search_values;

              if (search.length && search !== "") {

                let title = location.title.toLowerCase().includes(search);
                let place_id = location.place_id.toLowerCase().includes(search);
                let info =  JSON.parse(location.full_info)
                let addr = info.formatted_address.toLowerCase().includes(search);

                if (title || place_id || addr) {
                  filtered_locations.push(location);
                  //return location;
                }
              }
              else {
                filtered_locations = locations;
              }
            });
          }

          clearMarkers();
          renderLocationsOnMap(filtered_locations, locations);
          renderListItem(filtered_locations);
          infoWindow.close();
        })

        $('body', context).on('click', '.location-category-filter', function () {
          filtered_locations = [];
          //scrollToMap();
          let search_result = [];
          let locations = drupalSettings.locations !== undefined ? drupalSettings.locations : [];
          let has_selected_category = [];
          $('.location-category-filter').each(function () {
            if ($(this).is(':checked')) {
              has_selected_category.push(true);
              let selected_category = $(this).attr('id').toLowerCase();
              search_result.push(locations.filter(function (location) {
                  //location = location[0];
                  let location_category = location.category.toLowerCase();
                  return location_category.includes(selected_category) ? location : false;
                })
              );
            }
          });


          if (search_result && has_selected_category.length) {
            search_result.forEach(function (item) {
              item.map(function (element) {
                filtered_locations.push(element);
              })
            })
          }
          if (!has_selected_category.length) {
            filtered_locations = locations;
          }
          clearMarkers();
          renderLocationsOnMap(filtered_locations);
          renderListItem(filtered_locations);
        })

        /**
         * Open infoWindow on list item click.
         */
        $('body', context).on('click', '.place-wrapper', function () {
          clickPlaceItemList($(this));
          $(this).addClass('active').siblings().removeClass('active');
        })

        /**
         * Render filters
         */
        function renderCategories() {

          if (drupalSettings.icons !== undefined) {
            let icons = drupalSettings.icons;

            let $categories = `<ul class="d-flex justify-content-evenly">`;
            for (const [key, value] of Object.entries(icons)) {

              let translated_key = value.translation;
              let title = value.title;
              let file = value.file;
              $categories += `<li class="d-flex flex-column justify-content-center">
<label for="${title}"><img width="25px" height="auto" src="${file}"/><span>${translated_key}</span>
</label>

<input type="checkbox" class="location-category-filter" id="${title}" name="${title}"/></li>`;
            }
            $categories += `</ul>`;

            $('.location-filters-wrapper').append($categories);
          }
        }


        /**
         * Click list item.
         * @param element
         */
        function clickPlaceItemList(element) {
          let title = element.data('place-title');
          let place_id = element.data('place-id');
          let clicked = locations.filter(function (item) {
            if (place_id === item.place_id) {
              return item;
            }
          })

          infoWindow.close();

          let clicked_marker = markers.filter(function (item) {
            if (title === item.marker.title) {
              return item;
            }
          })
          let marker = clicked_marker[0].marker;
          infoWindow.setContent(renderInfo(clicked[0]));
          infoWindow.open(map, marker);

          google.maps.event.trigger(marker, 'click', {
              latLng: new google.maps.LatLng(0, 0),
            },
            map.panTo(marker.getPosition()),
          );
          setTimeout(function () {
            map.setZoom(15);
            //markerCluster.clear();
          }, 100);
        }

        function getLocationParam() {
          let title = '';
          let url = window.location.href;
          let url_parts = url.split('?');
          let params = url_parts[1];
          if (params !== undefined) {
            let params_parts = params.split('&');
            params_parts.map(function (item) {
              let param = item.split('=');
              if (param[0] === 'location') {
                title = param[1];
              }
            });
          }
          return title;
        }

        // function scrollToMap() {
        //   let target = $('#map');
        //   $('html,body').animate({
        //     scrollTop: target.offset().top
        //   }, 1000);
        // }
        function openSharedPlace(place) {
          let title = getLocationParam();
          if (decodeURI(title) === place.title) {
            //scrollToMap();
            markers.map(function (marker) {
              if (marker.marker.title === place.title) {
                google.maps.event.trigger(marker.marker, 'click', {
                    latLng: new google.maps.LatLng(0, 0),
                  },
                  map.panTo(marker.marker.getPosition()),
                );
                setTimeout(function () {
                  map.setZoom(18);
                }, 100)
              }
            });
          }
        }

        /**
         * Re
         */
        function requestUserPosition() {
          if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
              (position) => {
                initialize(position.coords.latitude, position.coords.longitude, 15);
              },
              () => {
                setTimeout(function () {
                  initialize();
                }, 100);
                if (use_datalayer) {
                  DataLayer.push({
                    'event': 'GAEvent',
                    'event_category': 'Map',
                    'event_action': 'Sucess',
                    'event_label': 'AllowGeolocation',
                    'interaction': 'true',
                    'component_name': 'Allow Geolocation Button',
                    'user_id_phone_hash': phone_hash
                  });
                }
                //handleLocationError(true, infoWindow, map.getCenter());
              }
            );
          } else {
            setTimeout(function () {
              initialize();
            }, 100);
            if (use_datalayer) {
              DataLayer.push({
                'event': 'GAEvent',
                'event_category': 'Map',
                'event_action': 'Decline',
                'event_label': 'DeclineGeolocation',
                'interaction': 'true',
                'component_name': 'Decline Geolocation Button',
                'user_id_phone_hash': phone_hash
              });
            }
            // Browser doesn't support Geolocation
            handleLocationError(false, infoWindow, map.getCenter());
          }
        }

        function handleLocationError(browserHasGeolocation, infoWindow, pos) {
          infoWindow.setPosition(pos);
          infoWindow.setContent(
            browserHasGeolocation
              ? "Error: The Geolocation service failed."
              : "Error: Your browser doesn't support geolocation."
          );
          infoWindow.open(map);
        }
      })

    }
  }

  Drupal.behaviors.share_button = {
    attach: function (context, settings) {

      $('#shareModal').on('shown.bs.modal', function (e) {
        var placeTitle = $(e.relatedTarget).data('place-title');
        $('#share-link').val(window.location.origin + window.location.pathname + '?location=' + placeTitle);
      });

      const maps = once('copy', '.share-container button#button-addon2', context);
      maps.forEach(function () {
        // Get the text field
        $('button#button-addon2').on('click', function () {
          var copyText = document.getElementById("share-link");

          // Select the text field
          copyText.select();
          copyText.setSelectionRange(0, 99999); // For mobile devices

          // Copy the text inside the text field
          navigator.clipboard.writeText(copyText.value);

          // Alert the copied text
          $(this).addClass('copied').val(Drupal.t('Copied')).find('.label').text(Drupal.t('Copied'));
          // DataLayer.push({
          //   'event': 'GAEvent',
          //   'event_category': 'Content',
          //   'event_action': 'Content',
          //   'event_label': "CopyLink", // example: Know More
          //   'interaction': 'true',
          //   'component_name': 'CopyLinkButton', // example:knowmore_button
          //   'element_text': 'Copy', // example: Know More
          //   'user_id_phone_hash': phone_hash
          // });
        });
      });

      const maps2 = once('copy2', '.share-container button#share-button', context);
      maps2.forEach(function () {
        // DataLayer.push({
        //   'event': 'GAEvent',
        //   'event_category': 'Content',
        //   'event_action': 'Content',
        //   'event_label': "ShareLink", // example: Know More
        //   'interaction': 'true',
        //   'component_name': 'ShareLinkButton', // example:knowmore_button
        //   'element_text': 'Share', // example: Know More
        //   'user_id_phone_hash': phone_hash
        // });

        $('button#share-button').on('click', function () {
          if (navigator.share) {
            var copyText = $("#share-link");
            navigator.share({
              title: Drupal.t('Corona.de shared content title.'),
              text: Drupal.t('Corona.de shared content text.'),
              url: copyText.val(),
            })
              .then(() => console.log('Successful share'))
              .catch((error) => console.log('Error sharing', error));
          } else {
            $('.share-container button#share-button').hide().before(Drupal.t('<p>Share not supported on this browser, do it the old way</p>'));
            //console.log('Share not supported on this browser, do it the old way.');
          }
        });
      })

      let switchButtons = once('switchButton', '.switch-button', context);
      switchButtons.forEach(function (button) {
        $(button).on('click', function () {
          $('.map-container').removeClass('d-none');
          $('.list-column').removeClass('d-none');
          let target = $(this).data('target');
          $(this).data('target', $(this).data('target') === 'map' ? 'list' : 'map');
          target === 'map' ? $('.map-container').addClass('d-none') : $('.list-column').addClass('d-none');
        });
      });

    }
  }


})(jQuery, Drupal, drupalSettings, once);
