<?php

namespace Drupal\abuseipdb\Controller;

use Drupal\Core\Controller\ControllerBase;
use GuzzleHttp\Client;

/**
 * Handles sending and receiving requests to the AbuseIPDB API.
 */
class Request extends ControllerBase {

  /**
   * The Guzzle client.
   *
   * @var GuzzleHttp\Client
   */
  private $client;

  /**
   * {@inheritdoc}
   */
  public function __construct() {
    $this->client = new Client([
      'base_uri' => 'https://api.abuseipdb.com/api/v2/',
    ]);
  }

  /**
   * Delivers a report to AbuseIPDB API.
   *
   * @param string $api_key
   *   The API key for the AbuseIPDB account.
   * @param string $ip
   *   The IP which is being checked or reported.
   * @param array $categories
   *   Categories for a report. Will be a list of integers.
   * @param string $comment
   *   Optional comment for report.
   *
   * @return Guzzle
   *   Guzzle class object.
   */
  public function report(string $api_key, string $ip, array $categories, string $comment = '') {
    $query = [
      'categories' => implode(',', $categories),
      'comment' => $comment,
      'ip' => $ip,
    ];

    $headers = [
      'Accept' => 'application/json',
      'Key' => $api_key,
    ];

    $res = $this->client->request(
      'POST',
      'report',
      ['headers' => $headers, 'query' => $query, 'http_errors' => FALSE]
    );

    return $res;
  }

  /**
   * Query the AbuseIPDB API for IP reports.
   *
   * @param string $api_key
   *   The API key for the AbuseIPDB account.
   * @param string $ip
   *   The IP which is being checked or reported.
   * @param int $days
   *   The number of days prior to return any reports of abuse.
   *
   * @return Guzzle
   *   Response property after request has been executed.
   */
  public function check(string $api_key, string $ip, int $days = NULL) {
    $query = [
      'ipAddress' => $ip,
      'maxAgeInDays' => $days,
    ];

    $headers = [
      'Accept' => 'application/json',
      'Key' => $api_key,
    ];

    $res = $this->client->request(
      'GET',
      'check',
      ['headers' => $headers, 'query' => $query, 'http_errors' => FALSE]
    );

    return $res;
  }

}
