<?php

namespace Drupal\abuseipdb\EventSubscriber;

use Drupal\abuseipdb\Reporter;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Checks list of abusive paths and bans the IPs which land there.
 */
class Terminate implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Current path.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  protected $pathCurrent;

  /**
   * Path matcher.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * Reporter.
   *
   * @var \Drupal\abuseipdb\Reporter
   */
  protected $reporter;

  /**
   * Request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   Current user.
   * @param \Drupal\Core\Database\Connection $database
   *   Database.
   * @param \Drupal\Core\Path\CurrentPathStack $path_current
   *   Current path.
   * @param \Drupal\Core\Path\PathMatcherInterface $path_matcher
   *   Path matcher.
   * @param \Drupal\abuseipdb\Reporter $reporter
   *   AbuseIPDB Reporter.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   Request stack.
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountProxyInterface $current_user, Connection $database, CurrentPathStack $path_current, PathMatcherInterface $path_matcher, Reporter $reporter, RequestStack $request_stack) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->database = $database;
    $this->pathCurrent = $path_current;
    $this->pathMatcher = $path_matcher;
    $this->reporter = $reporter;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[KernelEvents::TERMINATE][] = ['pathBlacklist'];
    return $events;
  }

  /**
   * {@inheritdoc}
   */
  public function pathBlacklist() {
    $abuseIpDbSettings = $this->configFactory->get('abuseipdb.settings');
    $anonymousOnly = $abuseIpDbSettings->get('abuseipdb.anonymous_only');

    // If module is only set to check anonymous traffic then
    // skip authenticated users. OR if the shutdown mode is enabled.
    if ($anonymousOnly == 1 && $this->currentUser->isAuthenticated() || $this->reporter->isShutdownMode()) {
      return;
    }

    $blacklist = $abuseIpDbSettings->get('abuseipdb.paths');

    // If there are no paths in the blacklist, return.
    if (empty($blacklist)) {
      return;
    }

    $current_path = $this->pathCurrent->getPath();
    $match = $this->pathMatcher->matchPath(strtolower($current_path), strtolower($blacklist));

    if ($match) {
      $request = $this->requestStack->getCurrentRequest();
      $ip = $request->getClientIp();
      $report_params = [];
      $report_params['ip'] = $ip;
      $report_params['categories'] = [21];
      $report_params['comment'] = $this->t('[Drupal AbuseIPDB module] Request path is blacklisted. @path', [
        '@path' => $current_path,
      ])->render();
      $this->reporter->setSettings($report_params);
      $this->reporter->report();

      $ban_ip = $abuseIpDbSettings->get('abuseipdb.paths_ban_ip');

      if ($ban_ip) {
        $this->reporter->ban();
      }
    }
  }

}
