<?php

namespace Drupal\abuseipdb;

use Drupal\abuseipdb\Controller\Request;
use Drupal\ban\BanIpManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;

/**
 * Handles the reporting of IP addresses.
 */
class Reporter {

  /**
   * The API key for the AbuseIPDB API.
   *
   * @var string
   */
  protected $apiKey;

  /**
   * The Ban Manager for the Ban module.
   *
   * @var \Drupal\ban\BanIpManager
   */
  protected $banManager;

  /**
   * The categories of abuse to report.
   *
   * @var array
   */
  protected $categories;

  /**
   * The CIDR of the IP address to report.
   *
   * @var string
   */
  protected $cidr;

  /**
   * The comment to include with the report.
   *
   * @var string
   */
  protected $comment = '';

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Database.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The number of days back to query for abuse reports.
   *
   * @var int
   */
  protected $days = 30;

  /**
   * The IP address to report.
   *
   * @var string
   */
  protected $ip;

  /**
   * The REST API request controller.
   *
   * @var \Drupal\abuseipdb\Controller\Request
   */
  protected $request;

  /**
   * The response from the AbuseIPDB API.
   *
   * @var \GuzzleHttp\Psr7\Response
   */
  protected $response = NULL;

  /**
   * The status of the shutdown setting.
   *
   * @var bool
   */
  protected $shutdown = FALSE;

  /**
   * Constructor.
   *
   * @param \Drupal\ban\BanIpManagerInterface $ban_manager
   *   Ban Manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param \Drupal\Core\Database\Connection $database
   *   Database.
   */
  public function __construct(BanIpManagerInterface $ban_manager, ConfigFactoryInterface $config_factory, Connection $database) {
    $this->banManager = $ban_manager;
    $this->configFactory = $config_factory;
    $this->database = $database;

    $abuseIpDbSettings = $this->configFactory->get('abuseipdb.settings');
    (bool) $this->shutdown = $abuseIpDbSettings->get('abuseipdb.shutdown');
  }

  /**
   * Initialize values which will be passed to a report request.
   *
   * @param array $options
   *   Report parameters array to set the class properties.
   */
  public function setSettings(array $options = []) {
    foreach ($options as $key => $value) {
      $this->{$key} = $value;
    }

    // Set the REST API request controller.
    $this->request = new Request();

    // Set API key.
    $abuseIpDbSettings = $this->configFactory->get('abuseipdb.settings');
    $api_key = $abuseIpDbSettings->get('abuseipdb.api_key');
    $this->setApiKey($api_key);

    // Remove categories which are unused.
    if ($this->categories) {
      $this->removeEmptyCategories($this->categories);
    }
  }

  /**
   * Ban current IP address.
   */
  public function ban() {
    $this->banManager->banIp($this->ip);
  }

  /**
   * Check if current IP is banned.
   */
  public function isBanned() {
    return $this->banManager->isBanned($this->ip);
  }

  /**
   * Check the current IP address.
   */
  public function check() {
    $this->response = $this->request->check($this->apiKey, $this->ip, $this->days);
  }

  /**
   * Report the current IP address.
   */
  public function report() {
    $this->response = $this->request->report($this->apiKey, $this->ip, $this->categories, $this->comment);
  }

  /**
   * Check if the current IP is abusive based on AbuseIPDB reports.
   *
   * @return bool
   *   True if the IP is abusive and False if the IP is not.
   */
  public function isAbusive() {
    $body = $this->getResponseBody();
    $body_array = json_decode($body);
    $abuseIpDbSettings = $this->configFactory->get('abuseipdb.settings');
    $abuseConfidenceScore = $abuseIpDbSettings->get('abuseipdb.abuse_confidence_score');

    if ($body_array->data->abuseConfidenceScore >= (int) $abuseConfidenceScore) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Return the AbuseIPDB Key.
   *
   * @return string
   *   Return the key.
   */
  public function getApiKey() {
    return $this->apiKey;
  }

  /**
   * Set the current AbuseIPDB Key.
   */
  public function setApiKey(string $api_key) {
    $this->apiKey = $api_key;
  }

  /**
   * Get the available categories of abuse.
   *
   * @return array
   *   List of integers which correspond to the abuse categories.
   */
  public function getCategories() {
    return $this->categories;
  }

  /**
   * Utility function to remove categories before reporting to AbuseIPDB.
   */
  public function removeEmptyCategories(array &$categories = []) {
    rsort($categories);
    while (count($categories) > 0 && $categories[count($categories) - 1] == 0) {
      array_pop($categories);
    }
  }

  /**
   * Sets the categories for this report.
   */
  public function setCategories(array $categories) {
    $this->removeEmptyCategories($categories);
    $this->categories = $categories;
  }

  /**
   * Return the report comment.
   *
   * @return string
   *   The comment.
   */
  public function getComment() {
    return $this->comment;
  }

  /**
   * Set the report comment.
   */
  public function setComment(string $comment) {
    $this->comment = $comment;
  }

  /**
   * Get the number of days back to query for abuse reports.
   *
   * @return int
   *   The number of days.
   */
  public function getDays() {
    return $this->days;
  }

  /**
   * Set the number of days back to query for abuse reports.
   */
  public function setDays(int $days) {
    $this->days = $days;
  }

  /**
   * Get the report IP address.
   *
   * @return string
   *   The IP address.
   */
  public function getIp() {
    return $this->ip;
  }

  /**
   * Set the report IP address.
   */
  public function setIp(string $ip) {
    $this->ip = $ip;
  }

  /**
   * Return the response Body from a Report request.
   *
   * @return string
   *   The response body from AbuseIPDB API.
   */
  public function getResponseBody() {
    return ($this->response) ? $this->response->getBody() : FALSE;
  }

  /**
   * Return the HTTP Status Code from a Report Request.
   *
   * @return int
   *   The HTTP response code from AbuseIPDB API.
   */
  public function getResponseStatusCode() {
    return ($this->response) ? $this->response->getStatusCode() : FALSE;
  }

  /**
   * Return the status of the shutdown setting.
   *
   * @return bool
   *   The status of the shutdown setting.
   */
  public function isShutdownMode(): bool {
    return $this->shutdown;
  }

  /**
   * Set the status of the shutdown setting.
   *
   * @param bool $shutdown
   *   The status of the shutdown setting.
   */
  public function setShutdownMode(bool $shutdown) {
    $this->shutdown = $shutdown;
  }

}
