<?php

namespace Drupal\abusive_traffic\Drush\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use GuzzleHttp\Client;
use League\OAuth2\Client\Provider\GenericProvider;
use phpseclib3\Net\SFTP;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 */
final class AbusiveTrafficCommands extends DrushCommands {

  /**
   * Access token used to access Acquia Cloud API endpoints.
   *
   * @var string
   */
  private string $accessToken;

  /**
   * Acquia application UUID to pull log files for.
   *
   * @var string
   */
  private string $acquiaApplicationUuid;

  /**
   * The id of PROD environment.
   *
   * @var string
   */
  private string $prodId;

  /**
   * Generic Provider used to create API requests.
   *
   * @var \League\OAuth2\Client\Provider\GenericProvider
   */
  private GenericProvider $provider;

  /**
   * GuzzleHttp client used to run API requests.
   *
   * @var \GuzzleHttp\Client
   */
  private Client $client;

  /**
   * Constructs an AbusiveTrafficCommands object.
   */
  public function __construct(
    private ConfigFactoryInterface $configFactory,
    private MailManagerInterface $mailManager,
    private FileSystemInterface $fileSystem,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.mail'),
      $container->get('file_system'),
    );
  }

  /**
   * Setup settings that are needed to connect to Acquia API.
   */
  private function initAcquiaApiClientAndProvider() {
    // https://docs.acquia.com/secrets
    $acquiaClientId = $this->configFactory->get('abusive_traffic_acquia_client_id')->get('key');
    $acquiaClientSecret = $this->configFactory->get('abusive_traffic_acquia_client_secret')->get('key');
    $this->acquiaApplicationUuid = $this->configFactory->get('abusive_traffic_acquia_application_uuid')->get('key') ?? '';

    // Get a client object to use for API calls.
    $this->client = new Client(['http_errors' => FALSE]);
    $this->provider = new GenericProvider([
      'clientId' => $acquiaClientId,
      'clientSecret' => $acquiaClientSecret,
      'urlAuthorize' => '',
      'urlAccessToken' => 'https://accounts.acquia.com/api/auth/oauth/token',
      'urlResourceOwnerDetails' => '',
    ]);

    // Get an access token using the client credentials grant.
    $this->accessToken = $this->provider->getAccessToken('client_credentials');
  }

  /**
   * Helper function to set the prodId that you'll need.
   */
  private function setupAcquiaIds() {

    // Generate a request object using the access token.
    $request = $this->provider->getAuthenticatedRequest(
      'GET',
      'https://cloud.acquia.com/api/applications/' . $this->acquiaApplicationUuid . '/environments',
      $this->accessToken
    );

    $response = $this->client->send($request);
    $responseBody = json_decode($response->getBody(), TRUE);

    // Figure out the applicationID for the prod environment.
    foreach ($responseBody['_embedded']['items'] as $environmentData) {
      // Using dev for testing only.
      if ($environmentData['name'] === 'prod') {
        $this->prodId = $environmentData['id'];
      }
    }
  }

  /**
   * Generate Acquia log file.
   */
  #[CLI\Command(name: 'abusive_traffic:generate-log', aliases: ['atgen'])]
  #[CLI\Usage(name: 'abusive_traffic:generate-log atgen', description: 'Generate Acquia log file.')]
  public function generateAcquiaLogFile() {
    $this->initAcquiaApiClientAndProvider();
    $this->setupAcquiaIds();

    $now = time();
    $dateFormat = 'Y-m-d\TH:i:s\+00:00';
    $toTime = date($dateFormat, $now);
    $secondsInOneHour = 60 * 60;
    $fromTime = date($dateFormat, $now - $secondsInOneHour);
    $method = 'POST';
    $url = 'https://cloud.acquia.com/api/environments/' . $this->prodId . '/logs/apache-access';
    $parameters = [
      'headers' => [
        'Host' => [
          'cloud.acquia.com',
        ],
        'Authorization' => [
          'Bearer ' . $this->accessToken,
        ],
      ],
      'form_params' => [
        'from' => $fromTime,
        'to' => $toTime,
      ],
    ];
    $this->client->request($method, $url, $parameters);
    $this->logger()->success(dt('Log file generated. It can be downloaded in ~5 minutes.'));
  }

  /**
   * Get Acquia log file.
   */
  #[CLI\Command(name: 'abusive_traffic:get-log', aliases: ['atget'])]
  #[CLI\Usage(name: 'abusive_traffic:get-log atget', description: 'Get Acquia log file.')]
  public function getAcquiaLogFile() {
    $this->initAcquiaApiClientAndProvider();
    $this->setupAcquiaIds();

    $request = $this->provider->getAuthenticatedRequest(
      'GET',
      'https://cloud.acquia.com/api/environments/' . $this->prodId . '/logs/apache-access',
      $this->accessToken,
    );

    // Send the request.
    $response = $this->client->send($request);
    $responseBody = $response->getBody()->getContents();

    // Want the filename to include the hour to overwrite existing items.
    $logFileNameGz = 'abusive-traffic-apache-' . date('h') . '.log.gz';
    // Where to save the file.
    $privatePath = $this->fileSystem->realpath('private://default/');
    $privatePath = $privatePath . '/';

    // Save file into drupal's private file location.
    file_put_contents($privatePath . $logFileNameGz, $responseBody);

    // For the gz file, need to read them and copy to a non-gz version.
    $bufferSize = 4096;
    $logFileName = str_replace('.gz', '', $logFileNameGz);
    $logFileGz = gzopen($privatePath . $logFileNameGz, 'rb');
    $logFile = fopen($privatePath . $logFileName, 'wb');

    // Keep repeating until the end of the input file.
    while (!gzeof($logFileGz)) {
      fwrite($logFile, gzread($logFileGz, $bufferSize));
    }

    // SFTP file only if setting is checked.
    if ($this->configFactory->get('abusive_traffic.settings')->get('forward_log_files')) {
      $sftpUsername = $this->configFactory->get('abusive_traffic_sftp_username')->get('key');
      $sftpPassword = $this->configFactory->get('abusive_traffic_sftp_password')->get('key');
      $sftpServer = $this->configFactory->get('abusive_traffic_sftp_server')->get('key');
      $sftpPath = $this->configFactory->get('abusive_traffic_sftp_path')->get('key');

      /** @var \phpseclib3\Net\SFTP $sftp */
      $sftp = new SFTP($sftpServer);

      // Returns true if successful. False if no connection.
      $sftpLoginResult = $sftp->login($sftpUsername, $sftpPassword);
      if (!$sftpLoginResult) {
        throw new \Exception('Could not connect to stfp.');
      }
      // Want to read file again.
      $logFile = fopen($privatePath . $logFileName, 'r');
      $sftp->put($sftpPath . $logFileName, $logFile, SFTP::SOURCE_LOCAL_FILE);
    }

    // Close the files once they are done with.
    fclose($logFile);
    gzclose($logFileGz);

    // Start of parsing the log file.
    $data = [];
    if (($logFileHandle = fopen($privatePath . $logFileName, 'r')) !== FALSE) {
      while (($logEntry = fgets($logFileHandle)) !== FALSE) {
        // https://docs.acquia.com/cloud-platform/monitor/logs/apache-access/#parsing-the-log-file
        // phpcs:ignore
        // [$client_ip, $identd, $http_auth_user_name, $date, $time, $request, $http_status_code, $bytes, $referrer, $user_agent, $vhost, $host, $hosting_site, $pid, $request_time, $ip_addresses_tracked, $request_id, $location] = str_getcsv($log_entry, ' ');
        $clientIp = str_getcsv($logEntry, ' ')[0];
        if ($clientIp === '-') {
          // Client IP is missing.
          continue;
        }

        if (!isset($data[$clientIp])) {
          $data[$clientIp] = 0;
        }

        $data[$clientIp] += 1;
      }

      // Make sure we actually reached the end of the log file and didn't
      // actually fail early.
      if (feof($logFileHandle) !== TRUE) {
        $this->logger()->warning(dt('something went wrong'));
      }

      fclose($logFileHandle);
    }

    // If the data count is > threshold, need to send an email.
    $threshold = $this->configFactory->get('abusive_traffic.settings')->get('threshold');
    $ipsOverThreshold = array_filter($data, function ($ipCount) use ($threshold) {
      return $ipCount >= $threshold;
    });

    // Remove items in the ignore list.
    // This could be done ahead of time, but this way there is less
    // to loop through.
    $ignorelist = $this->configFactory->get('abusive_traffic.settings')->get('ignorelist');
    // Remove * from ignorelist items.
    $ignorelist = preg_replace('/\\*/', '', $ignorelist);

    // Thank you shield module for this code.
    $ignorelist = array_filter(array_map('trim', explode("\n", $ignorelist)));
    foreach ($ipsOverThreshold as $ipAddress => $ipAddressCount) {
      foreach ($ignorelist as $ipIgnored) {
        // Compare $ipAddress vs $ipIgnored.
        if (str_starts_with($ipAddress, $ipIgnored)) {
          // Remove $ipAddress from $ipsOverThreshold.
          unset($ipsOverThreshold[$ipAddress]);
          continue;
        }
      }
    }

    if ($ipsOverThreshold) {
      $to = $this->configFactory->get('abusive_traffic.settings')->get('emaillist');
      $params = [
        'subject' => 'Abusive Traffic: threshold exceeded',
        'message' => 'Will be set in hook_mail().',
        'ipAddresses' => $ipsOverThreshold,
      ];
      $this->mailManager->mail(
        'abusive_traffic',
        'threshold_exceeded',
        $to,
        'en',
        $params,
        NULL,
        TRUE
      );
    }

  }

  /**
   * List Acquia applications you have access to.
   */
  #[CLI\Command(name: 'abusive_traffic:list-applications', aliases: ['atlist'])]
  #[CLI\FieldLabels(labels: [
    'name' => 'Name',
    'uuid' => 'Application UUID for secrets',
  ])]
  #[CLI\DefaultTableFields(fields: ['name', 'uuid'])]
  #[CLI\FilterDefaultField(field: 'name')]

  #[CLI\Usage(name: 'abusive_traffic:list-applications atlist', description: 'List Acquia Applications.')]
  public function listAcquiaApplications() {
    $this->initAcquiaApiClientAndProvider();

    // Generate a request object using the access token.
    $request = $this->provider->getAuthenticatedRequest(
      'GET',
      'https://cloud.acquia.com/api/applications',
      $this->accessToken
    );
    $response = $this->client->send($request);
    $responseBody = json_decode($response->getBody(), TRUE);

    $rows = [];
    foreach ($responseBody['_embedded']['items'] as $application) {
      $rows[] = [
        'name' => $application['name'],
        'uuid' => $application['uuid'],
      ];
    }

    $this->logger()->success(dt('You need to save the UUID to abusive_traffic_acquia_application_uuid.key'));

    return new RowsOfFields($rows);

  }

}
