<?php

declare(strict_types=1);

namespace Drupal\abusive_traffic\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Abusive traffic settings for this site.
 */
final class AbusiveTrafficSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'abusive_traffic_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['abusive_traffic.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Check to make sure all secret config are set.
    $acquiaClientId = $this->configFactory->get('abusive_traffic_acquia_client_id')->get('key');
    $acquiaClientSecret = $this->configFactory->get('abusive_traffic_acquia_client_secret')->get('key');
    $acquiaApplicationUuid = $this->configFactory->get('abusive_traffic_acquia_application_uuid')->get('key');
    if (!$acquiaClientId) {
      $form['abusive_traffic_acquia_client_id'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">abusive_traffic_acquia_client_id.key is missing. See README.md for instructions to set this.</div>',
      ];
    }
    if (!$acquiaClientSecret) {
      $form['abusive_traffic_acquia_client_secret'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">abusive_traffic_acquia_client_secret.key is missing. See README.md for instructions to set this.</div>',
      ];
    }
    if (!$acquiaApplicationUuid) {
      $form['abusive_traffic_acquia_application_uuid'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">abusive_traffic_acquia_application_uuid.key is missing. See README.md for instructions to set this.</div>',
      ];
    }
    if ($this->config('abusive_traffic.settings')->get('forward_log_files')) {
      $sftp_username = $this->configFactory->get('abusive_traffic_sftp_username')->get('key');
      $sftp_password = $this->configFactory->get('abusive_traffic_sftp_password')->get('key');
      $sftp_server = $this->configFactory->get('abusive_traffic_sftp_server')->get('key');
      $sftp_path = $this->configFactory->get('abusive_traffic_sftp_path')->get('key');
      // Need to have all of the options. $sftp_path could be '' if you want
      // files sent to the root.
      if (!$sftp_username || !$sftp_password || !$sftp_server || !$sftp_path) {
        $form['abusive_traffic_acquia_application_sftp'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--error">SFTP keys are missing. See README.md for instructions to set this.</div>',
        ];
      }
    }

    $form['threshold'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Threshold of when to send email'),
      '#description' => $this->t("If 1 IP address is over this number, an email will be generated."),
      '#default_value' => $this->config('abusive_traffic.settings')->get('threshold') ?? 100,
    ];
    $form['ignorelist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('IP ignorelist'),
      '#description' => $this->t("Enter list of IP's for which shield should not be shown, one per line. You can use a * wildcard (ex: 127.0.0.* or 127.0.*)."),
      '#default_value' => $this->config('abusive_traffic.settings')->get('ignorelist'),
    ];
    $form['emaillist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Email List'),
      '#description' => $this->t("Comma separated list of emails that should be alerted when the threshold is exceeded."),
      '#default_value' => $this->config('abusive_traffic.settings')->get('emaillist'),
    ];
    $form['forward_log_files'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Forward Log Files'),
      '#description' => $this->t("SFTP forward the log files so you can parse the files."),
      '#default_value' => $this->config('abusive_traffic.settings')->get('forward_log_files'),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $thresholdValue = $form_state->getValue('threshold');

    if (!is_numeric($thresholdValue) || $thresholdValue < 1) {
      $form_state->setErrorByName(
        'message',
        $this->t('Threshold needs to be more than 1. 100 is a good start.'),
      );
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('abusive_traffic.settings')
      ->set('threshold', $form_state->getValue('threshold'))
      ->set('ignorelist', $form_state->getValue('ignorelist'))
      ->set('emaillist', $form_state->getValue('emaillist'))
      ->set('forward_log_files', $form_state->getValue('forward_log_files'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
