<?php

namespace Drupal\access_by_ref\Controller;

use Drupal\access_by_ref\Entity\Abrconfig;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides a listing of Example.
 */
class AbrconfigListBuilder extends ConfigEntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('Machine name');
    $header['label'] = $this->t('Label');
    $header['bundle'] = $this->t('Bundle');
    $header['field'] = $this->t('Field');
    $header['reference_type'] = $this->t('Reference type');
    $header['extra'] = $this->t('Extra');
    $header['rights_type'] = $this->t('Rights type');
    $header['rights_read'] = $this->t('Read?');
    $header['rights_update'] = $this->t('Update?');
    $header['rights_delete'] = $this->t('Delete?');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface|Abrconfig $entity): array {
    $row['id'] = $entity->id();
    $row['label'] = $entity->label();
    $row['bundle'] = $entity->getBundle();
    $row['field'] = $entity->getField();
    $row['reference_type'] = $entity->getReferencetype(TRUE);
    $row['extra'] = $entity->getExtra();
    $row['rights_type'] = $entity->getRightsType();
    $row['rights_read'] = $entity->getRightsRead() ? 'X' : '';
    $row['rights_update']  = $entity->getRightsUpdate() ? 'X' : '';
    $row['rights_delete']  = $entity->getRightsDelete() ? 'X' : '';

    // You probably want a few more properties here...
    return $row + parent::buildRow($entity);
  }

  /**
   * Create a markup render array for specified text with tag as prefix/suffix.
   *
   * @param string $tag
   *   Tag to wrap with.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|null $text
   *   Text to wrap. If null, just provide wrapper.
   *
   * @return array
   *   The render array.
   */
  protected function wrapText(string $tag, TranslatableMarkup|NULL $text = NULL): array {
    $wrapped = [
      '#prefix' => "<$tag>",
      '#suffix' => "</$tag>",
    ];
    if (!is_null($text)) {
      $wrapped['#markup'] = $text;
    }
    return $wrapped;
  }

  /**
   * Return a render array containing help text.
   *
   * @return array
   *   Array of markup definitions containing help text.
   */
  protected function helpText(): array {
    $help = [];
    $help['info'] = $this->wrapText('p', $this->t('Lightweight module that extends read, update or delete permissions to a user in the following cases:'));
    $help['info']['types'] = $this->wrapText('ol');
    $help['info']['types']['user'] = $this->wrapText('li', $this->t('<strong>User: </strong>The node <em>references the user</em>'));
    $help['info']['types']['user_email'] = $this->wrapText('li', $this->t("<strong>User's mail:</strong> The node <em>references the user's e-mail.</em>"));
    $help['info']['types']['profile'] = $this->wrapText('li', $this->t("<strong>Profile value:</strong> The node has a value in a specified field that <em>is the same as one in the user's profile.</em>"));
    $help['info']['types']['inherit'] = $this->wrapText('li', $this->t('<strong>Inherit from parent: </strong> The node <em>references a node</em> that the user has certain permissions on.'));
    $help['info']['applies'] = $this->wrapText('p', $this->t('In each case, the rule only applies to logged-in users with general permission to access nodes by reference, and only on the node types and field names set in the configuration page.'));
    $help['info']['chainable'] = $this->wrapText('p', $this->t('Permissions are <strong>chainable</strong>, but note that there is no protection against infinite loops, so some care is advised in configuration.'));
    $help['info']['permissions'] = $this->wrapText('p', $this->t('These accesses will only be available to a user with the Access By Reference permissions, so be sure to set those.'));
    $help['info']['usage'] = $this->wrapText('p', $this->t("<b>To Use:</b> Select the content type to be controlled, and then the type of access you want to grant. Choose the field that will contain the effective data or link. In case we are looking for matched values, such as the 'Profile Value', specify in the Extra Field the field in the User Profile that has to match."));
    return $help;
  }

  /**
   * {@inheritdoc}
   */
  public function render(): array {

    $build['description'] = $this->helpText();

    $build['table'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#title' => $this->getTitle(),
      '#rows' => [],
      '#empty' => $this->t('There are no @label yet.', ['@label' => $this->entityType->getPluralLabel()]),
      '#cache' => [
        'contexts' => $this->entityType->getListCacheContexts(),
        'tags' => $this->entityType->getListCacheTags(),
      ],
    ];
    foreach ($this->load() as $entity) {
      if ($row = $this->buildRow($entity)) {
        $build['table']['#rows'][$entity->id()] = $row;
      }
    }

    // Only add the pager if a limit is specified.
    if ($this->limit) {
      $build['pager'] = [
        '#type' => 'pager',
      ];
    }
    return $build;
  }

}
