<?php

namespace Drupal\access_by_ref\Form;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityFieldManager;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\field\FieldConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the Example add and edit forms.
 */
class AbrconfigForm extends EntityForm {

  /**
   * Sets up dependency injection properties.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entityTypeManager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityBundleManager
   *   The entity bundle manager.
   * @param \Drupal\Core\Entity\EntityFieldManager $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheRenderer
   *   The cache renderer.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    protected EntityTypeBundleInfoInterface $entityBundleManager,
    protected EntityFieldManager $entityFieldManager,
    protected CacheBackendInterface $cacheRenderer,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_field.manager'),
      $container->get('cache.render')
    );
  }

  /**
   * Get all bundles of specified type to pre-fill available options in form.
   *
   * @param string $type
   *   The type of bundle - defaults to 'node'.
   *
   * @return array
   *   Array of discovered bundles.
   */
  public function getBundlesList(string $type = 'node'): array {
    $bundles = $this->entityBundleManager->getBundleInfo($type);
    foreach ($bundles as $key => &$item) {
      $bundles[$key] = $item['label'];
    }
    return $bundles;
  }

  /**
   * Return static list of reference type supported by the module.
   *
   * @return array
   *   Array of reference types.
   */
  public function getReferenceTypesList(): array {
    return [
      'user' => $this->t('User referenced'),
      'user_mail' => $this->t("User's mail"),
      'shared' => $this->t('Profile value'),
      'inherit' => $this->t('Inherit from parent'),
    ];

  }

  /**
   * Get the field data for the content type.
   *
   * @param string $contentType
   *   The name of the content type.
   * @param string $bundle
   *   The name of the content bundle.
   * @param string $property
   *   The property we are interested in - label, type, handler or omni.
   */
  public function fieldDataFetch(string $contentType, string $bundle = 'node', string $property = 'label') {
    $fields = [];

    if (!empty($contentType)) {
      $fields = array_filter(
        $this->entityFieldManager->getFieldDefinitions($bundle, $contentType), function ($field_definition) {
          return $field_definition instanceof FieldConfigInterface;
        }
      );
    }

    switch ($property) {
      case 'label':
        foreach ($fields as $key => &$field) {
          $fields[$key] = $field->label();
        }
        break;

      case 'type':
        foreach ($fields as $key => &$field) {
          $fields[$key] = $field->getType();
        }
        break;

      case 'handler':
        foreach ($fields as $key => &$field) {
          $fields[$key] = $field->getSetting('handler');
        }
        break;

      case 'omni':
        foreach ($fields as $key => &$field) {
          $vals = [
            'handler' => $field->getSetting('handler'),
            'type' => $field->getType(),
            'label' => $field->label(),
          ];
          $fields[$key] = $vals;
        }
        break;

    }

    return $fields;
  }

  /**
   * Get a list of fields available for nodes, grouped by node bundle.
   *
   * @return array
   *   Array of available fields.
   */
  public function getFieldsList(): array {
    $fields = [];
    $bundles = $this->getBundlesList();

    foreach ($bundles as $key => $label) {
      $fields[$key] = $this->fieldDataFetch($key);
      unset($fields[$key]['body']);
    }

    return $fields;
  }

  /**
   * Get a list of fields available for user entities.
   *
   * @return array
   *   Array of available fields.
   */
  public function getUserFieldsList(): array {
    $fields = [];
    $fields['User fields'] = $this->fieldDataFetch('user', 'user', 'label');
    return $fields;
  }

  /**
   * Gets a static list of access right types (view, update, delete).
   *
   * @return array
   *   Array of access right types.
   */
  public function getRightsTypeList(): array {
    return [
      'view' => $this->t('View'),
      'update' => $this->t("Update"),
      'delete' => $this->t('Delete'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\access_by_ref\Entity\Abrconfig $example */
    $example = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $example->label(),
      '#description' => $this->t("Label for the abr config entry."),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $example->id(),
      '#machine_name' => [
        'exists' => [$this, 'exist'],
      ],
      '#disabled' => !$example->isNew(),
    ];

    $form['bundle'] = [
      '#type' => 'select',
      '#empty_option' => $this->t('- Select -'),
      '#options' => $this->getBundlesList(),
      '#title' => $this->t('Bundle'),
      '#default_value' => $example->getBundle(),
      '#description' => $this->t("Bundle of the abr config entry."),
      '#required' => TRUE,
    ];

    $form['field'] = [
      '#type' => 'select',
      '#empty_option' => $this->t('- Select -'),
      '#options' => $this->getFieldsList(),
      '#title' => $this->t('Field'),
      '#default_value' => $example->getField(),
      '#description' => $this->t("The field of the bundle."),
      '#required' => TRUE,
    ];

    $form['reference_type'] = [
      '#type' => 'select',
      '#empty_option' => $this->t('- Select -'),
      '#options' => $this->getReferenceTypesList(),
      '#title' => $this->t('Reference type'),
      '#default_value' => $example->getReferenceType(),
      '#description' => $this->t("Reference type of the abr config entry."),
      '#required' => TRUE,
    ];

    $form['extra'] = [
      '#type' => 'select',
      '#empty_option' => $this->t('- Select -'),
      '#options' => $this->getUserFieldsList(),
      '#title' => $this->t('Extra field'),
      '#default_value' => $example->getExtra(),
      '#description' => $this->t("In case of matching a node value with user 'profile value', specify here the user field that has to match."),
      '#required' => FALSE,
    ];

    $form['rights_type'] = [
      '#type' => 'select',
      '#empty_option' => $this->t('- Select -'),
      '#options' => $this->getRightsTypeList(),
      '#title' => $this->t('Rights type field'),
      '#default_value' => $example->getRightsType(),
      '#description' => $this->t("Select against which operation the 'parent' node / element needs to be checked."),
      '#required' => FALSE,
    ];

    $form['rights_read'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Assign read rights?'),
      '#default_value' => $example->getRightsRead(),
      '#required' => FALSE,
    ];

    $form['rights_update'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Assign update rights?'),
      '#default_value' => $example->getRightsUpdate(),
      '#required' => FALSE,
    ];

    $form['rights_delete'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Assign delete rights?'),
      '#default_value' => $example->getRightsDelete(),
      '#required' => FALSE,
    ];

    $form['#attached']['library'][] = 'access_by_ref/configform';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\access_by_ref\Entity\Abrconfig $example */
    $example = $this->entity;
    $status = $example->save();

    if ($status === SAVED_NEW) {
      $this->messenger()->addMessage($this->t('The %label configuration created.', [
        '%label' => $example->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('The %label configuration updated.', [
        '%label' => $example->label(),
      ]));
    }

    // Wipe the Twig PHP Storage cache. ABR needed.
    $this->cacheRenderer->deleteAll();

    $form_state->setRedirect('entity.abrconfig.collection');
    return $status;
  }

  /**
   * Helper function to check whether an Example configuration entity exists.
   */
  public function exist($id): bool {
    $entity = $this->entityTypeManager->getStorage('abrconfig')->getQuery()
      ->accessCheck(FALSE)
      ->condition('id', $id)
      ->execute();
    return (bool) $entity;
  }

}
