<?php

namespace Drupal\access_by_ref\Plugin\migrate\source;

use Drupal\migrate_drupal\Plugin\migrate\source\VariableMultiRow;

/**
 * Access by Reference Drupal 7 configuration source plugin.
 *
 * Converts Access by Reference Drupal 7 configuration by
 * splitting lines and converting each line into an array
 * that is compatible with the D7AccessByRef process plugin.
 *
 * @MigrateSource(
 *   id = "d7_access_by_ref",
 *   source_module = "access_by_ref"
 * )
 */
class D7AccessByRef extends VariableMultiRow {

  /**
   * {@inheritdoc}
   */
  public function initializeIterator(): \ArrayIterator {
    $variables = [];

    foreach (parent::initializeIterator() as $row) {
      $variable_name = $row['name'];
      $raw_value = $row['value'];

      if (is_string($raw_value) && preg_match('/^s:\d+:"/', $raw_value)) {
        $raw_value = unserialize($raw_value, ['allowed_classes' => FALSE]);
      }

      $lines = preg_split('/\r\n|\r|\n/', $raw_value);
      foreach ($lines as $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }

        [$bundle, $field] = array_map('trim', explode('|', $line, 2));

        if (!empty($bundle) && !empty($field)) {
          $variables[] = [
            'id' => $this->generateMachineName(implode('__', [$variable_name, $bundle, $field])),
            'variable_name' => $variable_name,
            'bundle' => $bundle,
            'field' => $field,
          ];
        }
      }
    }

    return new \ArrayIterator($variables);
  }

  /**
   * {@inheritdoc}
   */
  public function fields(): array {
    return [
      'id' => $this->t('Generated ID based on the variable name and value.'),
      'variable_name' => $this->t('The original Drupal 7 abr config variable name.'),
      'bundle' => $this->t('The configured bundle of the abr config entry.'),
      'field' => $this->t('The configured field the abr config entry.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getIds(): array {
    return [
      'id' => [
        'type' => 'string',
      ],
    ];
  }

  /**
   * Generate machine name bases on label value.
   *
   * @param string $label
   *   The label to base the machine name on.
   *
   * @return string
   *   The generated machine name.
   */
  protected function generateMachineName(string $label): string {
    $label = strtolower($label);
    $label = preg_replace('/[^a-z0-9_]+/', '_', $label);
    $label = trim($label, '_');
    $label = substr($label, 0, 64);
    return $label;
  }

}
