<?php

namespace Drupal\Tests\access_by_ref\Functional;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\Tests\taxonomy\Traits\TaxonomyTestTrait;
use Drupal\taxonomy\TermInterface;

/**
 * Test the "Profile value" reference type using taxonomy entity reference.
 *
 * @group access_by_ref
 */
class ProfileEntityReferenceValueTest extends AccessByRefTestBase {

  use TaxonomyTestTrait;

  /**
   * Taxonomy term for team red.
   *
   * @var \Drupal\taxonomy\TermInterface
   */
  protected TermInterface $teamRed;

  /**
   * Taxonomy term for team blue.
   *
   * @var \Drupal\taxonomy\TermInterface
   */
  protected TermInterface $teamBlue;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function setUp(): void {
    parent::setUp();

    // Create taxonomy term reference Team field on content type.
    FieldStorageConfig::create([
      'field_name' => 'field_team',
      'type' => 'entity_reference',
      'entity_type' => 'node',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_team',
      'entity_type' => 'node',
      'bundle' => 'abrpage',
      'label' => 'Team',
    ])->save();

    // Create an ABR page node owned by admin user, with team red taxonomy
    // reference.
    $this->abrNodeId = Node::create([
      'title' => 'Test node',
      'type' => 'abrpage',
      'uid' => $this->abrAdminUser->id(),
      'field_team' => $this->teamRed->id(),
    ])->save();
  }

  /**
   * Override user creation to add permissions.
   *
   * For Profile Values, this function needs to override the base class version
   * to add the "field_user_team" field, and provide values for it.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function createAbrUsers(): void {
    // Vocabulary needs to be present before user setup.
    $vocabulary = $this->createVocabulary(['name' => 'Team']);
    $this->teamRed = $this->createTerm($vocabulary, ['name' => 'Red']);
    $this->teamBlue = $this->createTerm($vocabulary, ['name' => 'Blue']);

    // Create taxonomy reference field on user.
    FieldStorageConfig::create([
      'field_name' => 'field_user_team',
      'type' => 'entity_reference',
      'entity_type' => 'user',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_user_team',
      'entity_type' => 'user',
      'bundle' => 'user',
      'label' => 'Team',
    ])->save();

    $this->abrAdminUser = $this->drupalCreateUser(['administer access_by_ref settings']);
    $this->abrAllowedUser = $this->drupalCreateUser(
      permissions: ['access node by reference'],
      values: ['field_user_team' => $this->teamRed->id()],
    );
    // User with abr permission, but on wrong team.
    $this->abrOtherUser = $this->drupalCreateUser(
      permissions: ['access node by reference'],
      values: ['field_user_team' => $this->teamBlue->id()],
    );
    // User on correct team, but without abr permission.
    $this->nonAbrUser = $this->drupalCreateUser(
      values: ['field_user_team' => $this->teamRed->id()],
    );
  }

  /**
   * Submit new abr "profile value" reference type.
   */
  protected function submitAccessByRefForm():void {
    $this->submitForm([
      'label' => 'Abr page access by profile value',
      'id' => 'abr_page_access',
      'bundle' => 'abrpage',
      'field' => 'field_team',
      'reference_type' => 'shared',
      'extra' => 'field_user_team',
      'rights_update' => TRUE,
    ], 'Save');
  }

}
