<?php

declare(strict_types=1);

namespace Drupal\account_portal;

use Symfony\Component\HttpFoundation\Request;

/**
 * Account portal utility class.
 */
class AccountPortalUtility {

  /**
   * Same as `::getRedirectUri()` but just the base url.
   */
  public static function getRedirectBaseUri(Request $request) {
    if ($redirectUri = self::getRedirectUri($request)) {
      $parts = parse_url($redirectUri);
      if (!$parts || !isset($parts['scheme']) || !isset($parts['host'])) {
        return NULL;
      }

      $url = vsprintf('%s://%s', [
        $parts['scheme'],
        $parts['host'],
      ]);

      if (isset($parts['port'])) {
        $url .= ':' . $parts['port'];
      }

      return $url;
    }

    return NULL;
  }

  /**
   * Get redirect uri from the request.
   *
   * At it is tried to extract the uri from the `request_uri`
   * query parameter of the request. Otherwise the referrer
   * is being used.
   */
  public static function getRedirectUri(Request $request) {
    if ($redirectUri = self::getQueryParam($request, 'redirect_uri')) {
      return $redirectUri;
    }

    if ($referrer = $request->headers->get('referrer')) {
      return $referrer;
    }

    $customRefererHeaderName = \Drupal::getContainer()->getParameter('account_portal.custom_referer_header');
    if ($customRefererHeaderName && $referrer = $request->headers->get($customRefererHeaderName)) {
      return $referrer;
    }

    return NULL;
  }

  /**
   * Get query param from request.
   *
   * Searches recursively in the `destination` param url.
   */
  protected static function getQueryParam(Request $request, string $name): ?string {
    if ($value = $request->query->get($name)) {
      return $value;
    }

    if ($destination = $request->query->get('destination')) {
      return self::getQueryParam(Request::create($destination), $name);
    }

    return NULL;
  }

}
