<?php

declare(strict_types=1);

namespace Drupal\account_portal\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\account_portal\Routing\AccountPortalPathResolver;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Listen to kernel events.
 */
class KernelEventSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly AccountPortalPathResolver $pathResolver,
    #[Autowire(param: 'account_portal.invalid_consumer_id_destination')]
    protected readonly string|null $invalidConsumerIdDestination,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => [
        ['onRequestSetConsumer', 9999],
      ],
    ];
  }

  /**
   * Handle request.
   *
   * Checks if the request has the referer cookie set and if so,
   * sets the client id for the current request so that the
   * consumer.negotiator picks it up instead of the default one.
   */
  public function onRequestSetConsumer(RequestEvent $event) {
    $request = $event->getRequest();

    // If the current request has a account portal prefix,
    // extract the consumer id from it and set it as a query param
    // on the current request, so the `consumer.negotiator` service
    // picks it up correctly.
    // If the request also has the X-Consumer-ID header set, the header
    // takes precedence over this client id.
    if ($consumerId = $this->pathResolver->getPortalConsumerId($request)) {

      // If destination is set in
      // `account_portal.invalid_consumer_id_destination` and the consumer id
      // is not resolvable to an actual consumer, redrect to the destination.
      if (!!$this->invalidConsumerIdDestination && !$this->validateConsumerId($consumerId)) {
        $params = [];
        // Set the redirect_uri from the current request on the redirect.
        // This might be preferable to allow the user to be redirected back
        // to the origin.
        if ($redirectUri = $request->query->get('redirect_uri')) {
          $params['redirect_uri'] = $redirectUri;
        }

        $event->setResponse(new RedirectResponse(
          Url::fromRoute($this->invalidConsumerIdDestination, [], ['query' => $params])->toString()
        ));
      }
      // Set consumer id otherwise.
      else {
        $request->query->set('consumerId', $consumerId);
      }

    }
  }

  /**
   * Make sure consumer id belongs to an existing consumer.
   */
  protected function validateConsumerId(string $consumerId) {
    $storage = $this->entityTypeManager->getStorage('consumer');
    $results = $storage->loadByProperties(['client_id' => $consumerId]);

    return !empty($results);
  }

}
