<?php

declare(strict_types=1);

namespace Drupal\account_portal\Routing;

use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\Request;

/**
 * Path resolver for account portal requests.
 */
class AccountPortalPathResolver {

  public function __construct(
    #[Autowire(param: 'account_portal.base_path')]
    protected readonly string $basePath,
  ) {}

  /**
   * Get full account portal path prefix.
   *
   * Returns NULL if not a account portal path.
   */
  public function getPortalPathPrefix(Request $request) {
    $matches = $this->getPortalPathMatch($request);
    if (!$matches) {
      return NULL;
    }

    return $matches[0];
  }

  /**
   * Get consumer id from account portal path prefix.
   *
   * Returns NULL if not a account portal path.
   */
  public function getPortalConsumerId(Request $request) {
    $matches = $this->getPortalPathMatch($request);
    if (!$matches) {
      return NULL;
    }

    return $matches[1];
  }

  /**
   * Get portal path match.
   *
   * This returns NULL if the request path is not prefixed
   * by the account portal path.
   * Otherwise an array is returned with the first element being
   * the full path prefix and the second item being the consumer id.
   */
  public function getPortalPathMatch(Request $request) {
    $path = $request->getPathInfo();

    $regexifiedBasePath = str_replace('/', '\/', $this->basePath);
    $regex = sprintf('/^%s\/([a-zA-Z0-9-_]+)/', $regexifiedBasePath);

    if (!preg_match($regex, $path, $matches)) {
      return NULL;
    }

    return $matches;
  }

}
