<?php

declare(strict_types=1);

namespace Drupal\Tests\account_portal\Unit;

use Drupal\account_portal\AccountPortalUtility;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\DependencyInjection\Container;

/**
 * Tests for AccountPortalUtility.
 */
class AccountPortalUtilityTest extends UnitTestCase {

  /**
   * The container.
   */
  protected Container $container;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->container = new Container();
    $this->container->setParameter('account_portal.custom_referer_header', 'X-Custom-Referer');

    \Drupal::setContainer($this->container);
  }

  /**
   * Tests getRedirectUri with redirect_uri parameter.
   */
  public function testGetRedirectUriFromRedirectUri() {
    $request = Request::create('https://example.com/redirect?redirect_uri=https://example.com/callback');
    $this->assertSame('https://example.com/callback', AccountPortalUtility::getRedirectUri($request));
  }

  /**
   * Tests getRedirectUri with nested destination parameter.
   */
  public function testGetRedirectUriFromNestedDestination() {
    $request = Request::create('https://example.com/redirect?destination=/path?redirect_uri=https://example.com/nested');
    $this->assertSame('https://example.com/nested', AccountPortalUtility::getRedirectUri($request));
  }

  /**
   * Tests getRedirectUri with referrer header.
   */
  public function testGetRedirectUriFromReferrer() {
    $request = Request::create('https://example.com/redirect');
    $request->headers->set('referrer', 'https://example.com/from-referrer');
    $this->assertSame('https://example.com/from-referrer', AccountPortalUtility::getRedirectUri($request));
  }

  /**
   * Tests getRedirectUri with custom referrer header.
   */
  public function testGetRedirectUriFromCustomReferrer() {
    $request = Request::create('https://example.com/redirect');
    $request->headers->set('X-Custom-Referer', 'https://example.com/from-custom-referrer');
    $this->assertSame('https://example.com/from-custom-referrer', AccountPortalUtility::getRedirectUri($request));
  }

  /**
   * Tests getRedirectUri with no parameters or headers.
   */
  public function testGetRedirectUriWithNoParams() {
    $request = Request::create('https://example.com/redirect');
    $this->assertNull(AccountPortalUtility::getRedirectUri($request));
  }

  /**
   * Tests getRedirectBaseUri with standard URL.
   */
  public function testGetRedirectBaseUriStandard() {
    $request = Request::create('https://example.com/redirect?redirect_uri=https://test.com/callback');
    $this->assertSame('https://test.com', AccountPortalUtility::getRedirectBaseUri($request));
  }

  /**
   * Tests getRedirectBaseUri with port number.
   */
  public function testGetRedirectBaseUriWithPort() {
    $request = Request::create('https://example.com/redirect?redirect_uri=https://test.com:8080/callback');
    $this->assertSame('https://test.com:8080', AccountPortalUtility::getRedirectBaseUri($request));
  }

  /**
   * Tests getRedirectBaseUri with no redirect URI.
   */
  public function testGetRedirectBaseUriWithNoRedirect() {
    $request = Request::create('https://example.com/redirect');
    $this->assertNull(AccountPortalUtility::getRedirectBaseUri($request));
  }

  /**
   * Tests getRedirectBaseUri with invalid URL.
   */
  public function testGetRedirectBaseUriWithInvalidUrl() {
    $request = Request::create('https://example.com/redirect?redirect_uri=invalid-url');
    $this->assertNull(AccountPortalUtility::getRedirectBaseUri($request));
  }

}
