<?php

declare(strict_types=1);

namespace Drupal\account_portal\PathProcessor;

use Drupal\Core\PathProcessor\InboundPathProcessorInterface;
use Drupal\Core\PathProcessor\OutboundPathProcessorInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\account_portal\Routing\AccountPortalPathResolver;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\Request;

/**
 * Path processor to rewrite account portal path prefixes.
 */
class AccountPortalPathProcessor implements InboundPathProcessorInterface, OutboundPathProcessorInterface {

  public function __construct(
    protected readonly AccountPortalPathResolver $pathResolver,
    #[Autowire(param: 'account_portal.base_path')]
    protected readonly string $basePath,
    #[Autowire(param: 'account_portal.routes')]
    protected readonly array $routes,
  ) {}

  /**
   * Process inbound links.
   *
   * Removes the complete account portal prefix from inbound paths
   * to have them correctly resolved by drupal.
   *
   * {@inheritdoc}
   */
  public function processInbound($path, Request $request) {
    // Removes the account portal path prefix from the current
    // path if set.
    if ($prefix = $this->pathResolver->getPortalPathPrefix($request)) {
      return substr($path, strlen($prefix));
    }

    return $path;
  }

  /**
   * Processes outbound links.
   *
   * If the current page is requested via an account-portal path
   * (see `account_portal.base_path`), add this exact path prefix to all
   * outgoing links, if those match a route in `account_portal.routes`
   * service parameters.
   *
   * {@inheritdoc}
   */
  public function processOutbound($path, &$options = [], ?Request $request = NULL, ?BubbleableMetadata $bubbleable_metadata = NULL) {
    // Skip paths that do not have a route nor a current request.
    if (!$request || !isset($options['route'])) {
      return $path;
    }

    /** @var \Symfony\Component\Routing\Route $route */
    $route = $options['route'];

    // Skip routes, that are not marked for account portal handling.
    if (!$route->getDefault('_account_portal')) {
      return $path;
    }

    // If current request is for portal path, use this prefix.
    if ($prefix = $this->pathResolver->getPortalPathPrefix($request)) {
      return $prefix . $path;
    }

    return $path;
  }

}
