<?php

namespace Drupal\acquia_contenthub_publisher\Libs;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;

/**
 * An implementation of the config syndication settings using drupal config.
 */
class ConfigSyndicationSettings implements ConfigSyndicationSettingsInterface {

  /**
   * The name of the drupal configuration.
   */
  private const CONFIG_NAME = 'acquia_contenthub_publisher.exclude_settings';

  /**
   * The related configuration object.
   *
   * @var \Drupal\Core\Config\Config
   */
  private Config $config;

  /**
   * The configuration values.
   *
   * @var array
   */
  private array $values;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * ConfigSyndicationSettings constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
    $this->config = $this->getEditableConfig();
    $this->values = $this->config->getRawData();
  }

  /**
   * {@inheritdoc}
   */
  public function isConfigSyndicationDisabled(): bool {
    return $this->getConfig('exclude_config_entities', FALSE);
  }

  /**
   * {@inheritdoc}
   */
  public function toggleConfigSyndication(bool $disabled): void {
    $this->setConfig('exclude_config_entities', $disabled);
  }

  /**
   * {@inheritdoc}
   */
  public function save(): void {
    $this->config->setData($this->values);
    $this->config->save();
  }

  /**
   * Returns a config value based on the provided key.
   *
   * @param string $key
   *   The identifier of the value.
   * @param mixed $default_value
   *   The value if the key had no corresponding value.
   *
   * @return mixed
   *   The value on the specified key or provided default value.
   */
  protected function getConfig(string $key, $default_value = NULL) {
    return $this->getImmutableConfig()->get($key) ?? $default_value;
  }

  /**
   * Sets a configuration value based on the identifier.
   *
   * @param string $key
   *   The key to set the value at.
   * @param mixed $value
   *   The value to set.
   */
  protected function setConfig(string $key, $value): void {
    $this->values[$key] = $value;
  }

  /**
   * Provides editable exclude settings.
   *
   * @return \Drupal\Core\Config\Config
   *   Config object.
   */
  protected function getEditableConfig(): Config {
    return $this->configFactory->getEditable(self::CONFIG_NAME);
  }

  /**
   * Provides non-editable exclude settings.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   Non-editable config object
   */
  protected function getImmutableConfig(): ImmutableConfig {
    return $this->configFactory->get(self::CONFIG_NAME);
  }

}
