<?php

namespace Drupal\acquia_contenthub\Commands;

use Drupal\acquia_contenthub\Client\CdfMetricsManager;
use Drupal\acquia_contenthub\Client\ClientFactory;
use Drupal\acquia_contenthub\ContentHubConnectionManager;
use Drupal\acquia_contenthub\Libs\Traits\ResponseCheckerTrait;
use Drupal\acquia_contenthub\Settings\ContentHubConfigurationInterface;
use Drush\Commands\DrushCommands;
use Symfony\Component\Validator\Exception\MissingOptionsException;

/**
 * Drush commands for Client related operations.
 *
 * @package Drupal\acquia_contenthub\Commands
 */
class AcquiaContentHubClientCommands extends DrushCommands {

  use ResponseCheckerTrait;

  /**
   * Default cloud filter prefix.
   *
   * @var string
   */
  const DEFAULT_FILTER = 'default_filter_';

  /**
   * CH configurations.
   *
   * @var \Drupal\acquia_contenthub\Settings\ContentHubConfigurationInterface
   */
  protected ContentHubConfigurationInterface $achConfigurations;

  /**
   * The client factory.
   *
   * @var \Drupal\acquia_contenthub\Client\ClientFactory
   */
  protected ClientFactory $clientFactory;

  /**
   * The contenthub client object.
   *
   * @var \Acquia\ContentHubClient\ContentHubClient|bool
   */
  protected $client;

  /**
   * CH connection manager.
   *
   * @var \Drupal\acquia_contenthub\ContentHubConnectionManager
   */
  protected ContentHubConnectionManager $connectionManager;

  /**
   * Content Hub CDF metrics manager service.
   *
   * @var \Drupal\acquia_contenthub\Client\CdfMetricsManager
   */
  protected CdfMetricsManager $metricsManager;

  /**
   * Constructor.
   *
   * @param \Drupal\acquia_contenthub\Settings\ContentHubConfigurationInterface $ach_configurations
   *   The configuration factory.
   * @param \Drupal\acquia_contenthub\Client\ClientFactory $client_factory
   *   ACH client factory.
   * @param \Drupal\acquia_contenthub\ContentHubConnectionManager $connection_manager
   *   CH connection manager.
   * @param \Drupal\acquia_contenthub\Client\CdfMetricsManager $metrics_manager
   *   Content Hub cdf metrics manager.
   */
  public function __construct(ContentHubConfigurationInterface $ach_configurations, ClientFactory $client_factory, ContentHubConnectionManager $connection_manager, CdfMetricsManager $metrics_manager) {
    $this->achConfigurations = $ach_configurations;
    $this->clientFactory = $client_factory;
    $this->connectionManager = $connection_manager;
    $this->metricsManager = $metrics_manager;
  }

  /**
   * Updates Client Details.
   *
   * Updates default filter name, if client update is successful.
   *
   * @command acquia:contenthub:client-update
   * @aliases ach-cu
   *
   * @option name
   *    The new client name.
   * @default name
   *
   * @usage acquia:contenthub:client-update --name new_name
   *   Updates the Acquia Content Hub Client Name.
   *
   * @throws \Exception
   */
  public function updateClient(): int {
    $this->client = $this->clientFactory->getClient();
    if (!$this->client) {
      $this->io()->error('Error while connecting to Content Hub. Make sure this site is registered to Content Hub.');
      return self::EXIT_FAILURE;
    }
    try {
      $this->updateClientName();
    }
    catch (\Exception $e) {
      $this->io()->error($e->getMessage());
      return self::EXIT_FAILURE;
    }

    return self::EXIT_SUCCESS;
  }

  /**
   * Updates Client Name.
   *
   * @throws \Exception
   *   Throws exception if client could not be updated.
   */
  protected function updateClientName(): void {
    $name = $this->input()->getOption('name');
    if (empty($name)) {
      throw new MissingOptionsException('Name option not passed, new client name is required to run client-update command ', []);
    }
    $ch_connection = $this->achConfigurations->getConnectionDetails();
    if ($ch_connection->isOverridden('client_name')) {
      $this->io()->warning('Client Name is manually changed in settings.php. Drush command can not override it');
      return;
    }
    $client_name = $this->achConfigurations->getConnectionDetails()->getClientName();
    $response = $this->client->updateClient($ch_connection->getClientUuid(), $name);
    if (!$this->isSuccessful($response)) {
      throw new \Exception(sprintf('Client update has failed: %s', $response->getBody()));
    }
    $ch_connection->setClientName($name);
    $this->io()->success('Client name changed successfully.');

    // Refresh client object with recently saved settings.
    $this->client = $this->clientFactory->getClient($this->achConfigurations->getSettings());
    $this->metricsManager->sendClientCdfUpdates($this->client);

    $old_default_filter_name = self::DEFAULT_FILTER . $client_name;
    $new_default_filter_name = self::DEFAULT_FILTER . $name;
    $this->updateDefaultFilter($old_default_filter_name, $new_default_filter_name);
  }

  /**
   * Updates default filter.
   *
   * @param string $old_default_filter_name
   *   Name of old default filter.
   * @param string $new_default_filter_name
   *   Name of a new default filter.
   */
  protected function updateDefaultFilter(string $old_default_filter_name, string $new_default_filter_name): void {
    $response = $this->connectionManager->updateFilterName($old_default_filter_name, $new_default_filter_name);

    if (!$response) {
      return;
    }
    if ($response['success']) {
      $this->io()->success('Default filter updated successfully.');
      return;
    }
    if (!$response['success'] && !empty($response['error']['message'])) {
      $this->io()->error(sprintf('Default filter update failed. Error: %s', $response['error']['message']));
    }
  }

}
