<?php

namespace Drupal\acquia_cookie_vary\EventSubscriber;

use Drupal\Core\Cache\CacheableResponseInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Conditionally adds the proper vary header to responses.
 */
class ResponseSubscriber implements EventSubscriberInterface {

  /**
   * A mapping of standard platform cookie names to request header values.
   */
  protected const PLATFORM_COOKIES = [
    'acquia_a' => 'X-Acquia-Cookie-A',
    'acquia_b' => 'X-Acquia-Cookie-B',
    'acquia_c' => 'X-Acquia-Cookie-C',
  ];

  /**
   * Service constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [KernelEvents::RESPONSE => ['onKernelResponse', 1000]];
  }

  /**
   * Invoked on kernel response events.
   */
  public function onKernelResponse(ResponseEvent $event) {
    $response = $event->getResponse();
    if ($response instanceof CacheableResponseInterface) {
      $config = $this->configFactory->get('acquia_cookie_vary.settings');

      $contexts = $response->getCacheableMetadata()->getCacheContexts();
      foreach (static::PLATFORM_COOKIES as $cookie => $header) {
        if (in_array("cookies:$cookie", $contexts, TRUE)) {
          $response->setVary($header, FALSE);
          if ($config->get('debug')) {
            $response->headers->set(
              $header,
              $event->getRequest()->cookies->get($cookie, ''),
            );
          }
        }
      }

      $custom_cookie = $config->get('custom_cookie');
      if (in_array("cookies:$custom_cookie", $contexts, TRUE)) {
        $response->setVary('X-Acquia-Cookie-Key', FALSE);
        $response->setVary('X-Acquia-Cookie-Value', FALSE);
        if ($config->get('debug')) {
          $response->headers->set(
            'X-Acquia-Cookie-Key',
            $custom_cookie,
          );
          $response->headers->set(
            'X-Acquia-Cookie-Value',
            $event->getRequest()->cookies->get($custom_cookie, ''),
          );
        }
      }
    }
  }

}
