<?php

namespace Drupal\Tests\acquia_cookie_vary\Kernel;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\KernelTests\KernelTestBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\HttpKernel\Kernel;

/**
 * Test cases for the response subscriber.
 *
 * @group acquia_cookie_vary
 */
class ResponseSubscriberTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['acquia_cookie_vary'];

  /**
   * The subject under test.
   *
   * @var \Drupal\acquia_cookie_vary\EventSubscriber\ResponseSubscriber
   */
  protected $instance;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['acquia_cookie_vary']);
    $this->instance = $this->container->get('acquia_cookie_vary.response_subscriber');
  }

  /**
   * Data provider for the response subscriber test.
   */
  public function variationProvider(): array {
    return [
      'no variation' => [[], [], [], [], []],
      'acquia_a' => [
        ['custom_cookie' => '', 'debug' => FALSE],
        ['acquia_a' => 'a'],
        ['cookies:acquia_a'],
        ['X-Acquia-Cookie-A'],
        [],
      ],
      'acquia_a (debug)' => [
        ['custom_cookie' => '', 'debug' => TRUE],
        ['acquia_a' => 'a'],
        ['cookies:acquia_a'],
        ['X-Acquia-Cookie-A'],
        ['x-acquia-cookie-a' => ['a']],
      ],
      'acquia_b' => [
        ['custom_cookie' => '', 'debug' => FALSE],
        ['acquia_a' => 'b'],
        ['cookies:acquia_b'],
        ['X-Acquia-Cookie-B'],
        [],
      ],
      'acquia_b (debug)' => [
        ['custom_cookie' => '', 'debug' => TRUE],
        ['acquia_b' => 'b'],
        ['cookies:acquia_b'],
        ['X-Acquia-Cookie-B'],
        ['x-acquia-cookie-b' => ['b']],
      ],
      'acquia_c' => [
        ['custom_cookie' => '', 'debug' => FALSE],
        ['acquia_c' => 'c'],
        ['cookies:acquia_c'],
        ['X-Acquia-Cookie-C'],
        [],
      ],
      'acquia_c (debug)' => [
        ['custom_cookie' => '', 'debug' => TRUE],
        ['acquia_c' => 'c'],
        ['cookies:acquia_c'],
        ['X-Acquia-Cookie-C'],
        ['x-acquia-cookie-c' => ['c']],
      ],
      'custom' => [
        ['custom_cookie' => 'test', 'debug' => FALSE],
        ['test' => 'value'],
        ['cookies:test'],
        ['X-Acquia-Cookie-Key', 'X-Acquia-Cookie-Value'],
        [],
      ],
      'custom (debug)' => [
        ['custom_cookie' => 'test', 'debug' => TRUE],
        ['test' => 'value'],
        ['cookies:test'],
        ['X-Acquia-Cookie-Key', 'X-Acquia-Cookie-Value'],
        ['x-acquia-cookie-key' => ['test'], 'x-acquia-cookie-value' => ['value']],
      ],
      'kitchen sink' => [
        ['custom_cookie' => 'test', 'debug' => FALSE],
        ['acquia_a' => 'a', 'acquia_b' => 'b', 'acquia_c' => 'c', 'test' => 'value'],
        ['cookies:acquia_a', 'cookies:acquia_b', 'cookies:acquia_c', 'cookies:test'],
        ['X-Acquia-Cookie-A', 'X-Acquia-Cookie-B', 'X-Acquia-Cookie-C', 'X-Acquia-Cookie-Key', 'X-Acquia-Cookie-Value'],
        [],
      ],
      'kitchen sink (debug)' => [
        ['custom_cookie' => 'test', 'debug' => TRUE],
        ['acquia_a' => 'a', 'acquia_b' => 'b', 'acquia_c' => 'c', 'test' => 'value'],
        ['cookies:acquia_a', 'cookies:acquia_b', 'cookies:acquia_c', 'cookies:test'],
        ['X-Acquia-Cookie-A', 'X-Acquia-Cookie-B', 'X-Acquia-Cookie-C', 'X-Acquia-Cookie-Key', 'X-Acquia-Cookie-Value'],
        [
          'x-acquia-cookie-a' => ['a'],
          'x-acquia-cookie-b' => ['b'],
          'x-acquia-cookie-c' => ['c'],
          'x-acquia-cookie-key' => ['test'],
          'x-acquia-cookie-value' => ['value'],
        ],
      ],
    ];
  }

  /**
   * Test case for the response subscriber.
   *
   * @param array $settings
   *   The settings to run the test with.
   * @param array $cookies
   *   The cookies to set in the request.
   * @param array $contexts
   *   The cache contexts to set in the response.
   * @param array $expected_vary
   *   The expected vary values.
   * @param array $expected_headers
   *   The expected header values.
   *
   * @dataProvider variationProvider
   */
  public function testResponseSubscriber(array $settings, array $cookies, array $contexts, array $expected_vary, array $expected_headers) {

    $this->config('acquia_cookie_vary.settings')
      ->setData($settings)
      ->save();

    $request = new Request();
    foreach ($cookies as $name => $value) {
      $request->cookies->set($name, $value);
    }

    $response = new CacheableResponse();
    $response->addCacheableDependency(
      CacheableMetadata::createFromRenderArray([
        '#cache' => [
          'contexts' => $contexts,
        ],
      ])
    );

    $event = new ResponseEvent(
      $this->createMock(Kernel::class),
      $request,
      HttpKernelInterface::MAIN_REQUEST,
      $response
    );

    $this->instance->onKernelResponse($event);

    // Ensure vary headers are set appropriately.
    $vary = $response->getVary();

    static::assertSame($expected_vary, $vary);

    // Clean up the headers so we can perform exact matching.
    $actual_headers = $response->headers->all();
    unset($actual_headers['date'], $actual_headers['cache-control'], $actual_headers['vary']);

    static::assertSame($expected_headers, $actual_headers);
  }

}
