<?php

declare(strict_types=1);

namespace Drupal\acquiadam_asset_import\Form;

use Drupal\media_acquiadam\Plugin\media\Source\AcquiadamAsset;
use Drupal\acquia_dam\Entity\MediaSourceField;
use Drupal\acquia_dam\Plugin\media\Source\Asset;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\media\MediaInterface;
use Drupal\media\MediaTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller of the bulk import configuration form.
 */
class BulkImportConfigForm extends ConfigFormBase {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Name of the current route.
   *
   * @var string
   */
  protected $currentRouteName;

  /**
   * The current active user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Acquia DAM client on behalf of the current user.
   *
   * @var \Drupal\acquia_dam\Client\AcquiaDamClientFactory
   */
  protected $userClientFactory;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Acquia DAM logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Acquia DAM asset import queue.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $assetImportQueue;

  /**
   * Acquia DAM media type resolver.
   *
   * @var \Drupal\acquia_dam\MediaTypeResolver
   */
  protected $mediaTypeResolver;

  /**
   * Store module-wide used constant as class attribute.
   *
   * @var string
   */
  protected $sourceFieldName;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->configFactory = $container->get('config.factory');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentRouteName = $container->get('current_route_match')->getRouteName();
    $instance->currentUser = $container->get('current_user');
    $instance->userClientFactory = $container->get('acquia_dam.client.factory');
    $instance->messenger = $container->get('messenger');
    $instance->logger = $container->get('logger.channel.acquia_dam');
    $instance->assetImportQueue = $container->get('queue')->get('acquia_dam_asset_import');
    $instance->mediaTypeResolver = $container->get('acquia_dam.media_type_resolver');
    $instance->sourceFieldName = MediaSourceField::SOURCE_FIELD_NAME;

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'acquiadam_asset_import_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['acquiadam_asset_import.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save and schedule'),
        '#button_type' => 'primary',
      ],
      'cancel' => [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => Url::fromRoute($this->currentRouteName),
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ],
    ];

    // Start by fetching the remote data.
    try {
      $response = $this->userClientFactory->getUserClient()->getCategories();
    }
    catch (\Exception $exception) {
      $this->messenger->addWarning('Something went wrong while gathering categories from the remote DAM service. Please contact the site administrator.');
      $this->logger->error($exception->getMessage());
      return $form;
    }

    // Bail out when no input source is available.
    if (!isset($response['total_count']) || $response['total_count'] < 1) {
      $form['content'] = [
        '#markup' => $this->t('No categories were found available for the currently authorized user account on the remote DAM service. Bulk importing media assets requires at least one category to be available to fetch assets from. Please ensure that at least one category exists and is accessible for the current user or <a href=":url">authenticate with a different user account</a>.', [
          ':url' => Url::fromRoute('entity.user.acquia_dam_auth', [
            'user' => $this->currentUser->id(),
          ])->toString(),
        ]),
      ];
      $form['actions']['submit']['#disabled'] = TRUE;

      return $form;
    }

    $categ_opts = [];
    foreach ($response['items'] as $item) {
      $categ_opts[$item['id']] = $item['name'];
    }

    $media_types = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    $dam_bundles = array_filter($media_types, static function (MediaTypeInterface $media_type) {
      return $media_type->getSource() instanceof Asset;
    });
    $media_acquiadam_bundles = array_filter($media_types, static function (MediaTypeInterface $media_type) {
      return $media_type->getSource() instanceof AcquiadamAsset;
    });

    // Bail out when no output target is available.
    if (count($dam_bundles) < 1) {
      $form['content'] = [
        '#markup' => $this->t('No DAM-capable media types found currently exist on this site. Bulk importing media assets requires at least one media type to be available to save fetched assets. <a href=":url">Check media types</a>', [
          ':url' => Url::fromRoute('entity.media_type.collection')->toString(),
        ]),
      ];
      $form['actions']['submit']['#disabled'] = TRUE;

      return $form;
    }

    $bundl_opts = [];
    foreach ($dam_bundles as $key => $value) {
      $bundl_opts[$key] = $value->label();
      $descriptions[$key] = $value->get('description');
    }

    // Start building up the form.
    $form['editor']['category_uuid'] = [
      '#type' => 'select',
      '#title' => $this->t('Source category'),
      '#description' => $this->t('List of categories in the remote DAM system available for the authorized user account. Please choose which of them the media assets should be imported from. When adding the same category multiple times, only the last row of them will be considered and saved.'),
      '#options' => $categ_opts,
      '#sort_options' => TRUE,
      '#empty_value' => '',
      '#empty_option' => $this->t('Please choose one…'),
    ];
    $form['editor']['activate_filtering'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Filter assets'),
      '#description' => $this->t('The website automatically detects the file type of each asset in remote DAM and does its best to guess which media type it should be assigned to. In the case of a category containing various types of assets, this filtering option allows you to import only those assets which would be assigned to the given media types.'),
      '#default_value' => $form_state->get('activate_filtering') ?? 0,
      '#states' => [
        'visible' => [
          ':input[name="category_uuid"]' => ['filled' => TRUE],
        ],
      ],
    ];
    $form['editor']['filter_media_bundles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Import only assets which would be assigned to these media types'),
      '#description' => $this->t('List of DAM-capable media types currently existing on this site. Set the filtration by specifying those media types that are allowed to receive assets during the import process. Any other assets within the defined category not fitting any allowed media type listed here will be skipped.'),
      '#options' => $bundl_opts,
      '#sort_options' => TRUE,
      '#multiple' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="activate_filtering"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['editor']['add_actions'] = [
      '#type' => 'actions',
    ];
    $form['editor']['add_actions']['add_button'] = [
      '#type' => 'submit',
      '#value' => $this->t('+ Add'),
      '#submit' => ['::addSubmitMethod'],
      '#ajax' => [
        'callback' => '::assignmentsCallback',
        'wrapper' => 'assignment-list-wrapper',
      ],
      '#button_type' => 'secondary',
      '#title' => $this->t('Add the current combination of source category and target media type to the list below.'),
      '#title_display' => 'attribute',
      '#states' => [
        'visible' => [
          ':input[name="category_uuid"]' => ['filled' => TRUE],
        ],
      ],
    ];

    // Assign descriptions to the checkboxes.
    foreach ($form['editor']['filter_media_bundles']['#options'] as $option_value => $option_label) {
      $form['editor']['filter_media_bundles'][$option_value]['#description'] = $descriptions[$option_value];
    }

    // Convert data stored in module config into a numerically keyed array.
    $config_data = [];
    foreach ((array) $this->configFactory->get('acquiadam_asset_import.settings')->get('categories') as $category_uuid => $bundles) {
      $config_data[] = [
        'category_uuid' => $category_uuid,
        'media_bundles' => $bundles,
      ];
    }

    // Utility variable to track the total number of rows currently rendered.
    $rows_count = $form_state->get('rows_count');
    if ($rows_count === NULL) {
      $form_state->set('rows_count', count($config_data));
      $rows_count = $form_state->get('rows_count');
    }

    // Utility variable to track the indexes of rows already deleted earlier.
    $rows_removed = $form_state->get('rows_removed');
    if ($rows_removed === NULL) {
      $form_state->set('rows_removed', []);
      $rows_removed = $form_state->get('rows_removed');
    }

    // Convert output of the <select> list into a more usable format.
    $bundles_selected = NULL;
    if ($form_state->getValue('activate_filtering') && $form_state->getValue('filter_media_bundles')) {
      foreach ($form_state->getValue('filter_media_bundles') as $key => $value) {
        if ($value !== 0) {
          $bundles_selected[] = $key;
        }
      }
    }

    $form['assignment_list'] = [
      '#type' => 'table',
      '#header' => [
        'category_info' => $this->t('From these categories…'),
        'media_bundles' => $this->t('…only these type of media'),
        'remove_button' => '',
      ],
      '#prefix' => '<div id="assignment-list-wrapper">',
      '#suffix' => '</div>',
      '#empty' => $this->t('No category has been selected yet.'),
    ];

    // Iterate through all the rows the form in its current state has.
    for ($i = 0; $i < $rows_count; $i++) {
      $placeholders = [];
      // Check if the row was once removed, then skip it.
      if (in_array($i, $rows_removed)) {
        continue;
      }

      // First try to load the latest data stored in previous run.
      if (isset($form_state->get('assignments')[$i])) {
        $placeholders[$i]['category_uuid'] = $form_state->get('assignments')[$i]['category_uuid'];
        $placeholders[$i]['media_bundles'] = $form_state->get('assignments')[$i]['media_bundles'];
      }
      // If form state is empty, then read data from the configuration.
      elseif (isset($config_data[$i])) {
        $placeholders[$i]['category_uuid'] = $config_data[$i]['category_uuid'];
        $placeholders[$i]['media_bundles'] = $config_data[$i]['media_bundles'];
      }
      // At this point probably the newest row is being rendered in the last
      // iteration, so read data from the editor.
      elseif ($i === $rows_count - 1) {
        $placeholders[$i]['category_uuid'] = $form_state->getValue('category_uuid');
        $placeholders[$i]['media_bundles'] = $bundles_selected;
      }
      // This case should not happen.
      else {
        $error_message = $this->t('Invalid case occurred.');
        $placeholders[$i]['category_uuid'] = $error_message;
        $placeholders[$i]['media_bundles'] = [$error_message];
      }

      $form['assignment_list'][$i]['category_info'] = [
        '#markup' => $this->userClientFactory->getUserClient()->convertCategoryUuidToName($placeholders[$i]['category_uuid']),
      ];

      if (is_array($placeholders[$i]['media_bundles']) && count($placeholders[$i]['media_bundles']) > 0) {
        $form['assignment_list'][$i]['media_bundles'] = [
          '#type' => 'html_tag',
          '#tag' => 'ul',
        ];

        foreach ((array) $placeholders[$i]['media_bundles'] as $key => $value) {
          if ($value !== 0) {
            $legacy_bundle = in_array($value, array_keys($media_acquiadam_bundles));
            $form['assignment_list'][$i]['media_bundles']['data'][] = [
              '#type' => 'html_tag',
              '#tag' => 'li',
              '#value' => $legacy_bundle ? $media_acquiadam_bundles[$value]->label() . ' ' . $this->t('(Import Disabled: Media type requires migration)') : $dam_bundles[$value]->label(),
            ];
          }
        };
      }
      else {
        $form['assignment_list'][$i]['media_bundles'] = [
          '#markup' => $this->t('All assets (no filtering)'),
        ];
      }

      $form['assignment_list'][$i]['remove_button']['data'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_button_id_' . $i,
        '#submit' => ['::removeSubmitMethod'],
        '#limit_validation_errors' => [
          ['assignment_list'],
        ],
        '#ajax' => [
          'callback' => '::assignmentsCallback',
          'wrapper' => 'assignment-list-wrapper',
          'disable-refocus' => TRUE,
        ],
        '#attributes' => [
          'class' => ['button--danger', 'js-form-submit'],
          'data-disable-refocus' => 'true',
        ],
      ];

      // Store current form data between callback invokes.
      $form_state->set(['assignments', $i], [
        'category_uuid' => $placeholders[$i]['category_uuid'],
        'media_bundles' => $placeholders[$i]['media_bundles'],
      ]);
    }

    return $form;
  }

  /**
   * Callback for both the "Add" & "Remove" Ajax-enabled buttons.
   *
   * Returns the updated assignment list.
   */
  public function assignmentsCallback(array &$form, FormStateInterface $form_state) {
    return $form['assignment_list'];
  }

  /**
   * Submit handler for the "+ Add" button.
   *
   * Updates the summary table and causes a form rebuild.
   */
  public function addSubmitMethod(array &$form, FormStateInterface $form_state) {
    $rows_count = $form_state->get('rows_count');
    $rows_count++;
    $form_state->set('rows_count', $rows_count);

    // Reset the Categories form.
    $form_state->set('activate_filtering', 0);
    $form_state->set('category_uuid', '');
    $form_state->setRebuild();
  }

  /**
   * Submit handler for the "Remove" buttons.
   *
   * Updates the summary table and causes a form rebuild.
   */
  public function removeSubmitMethod(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $index_to_remove = str_replace('remove_button_id_', '', $triggering_element['#name']);
    $assignment_list = $form_state->getValue('assignment_list', []);
    unset($assignment_list[$index_to_remove]);
    $form_state->setValue('assignment_list', $assignment_list);

    // Update the list of removed rows.
    $rows_removed = $form_state->get('rows_removed');
    $rows_removed[] = $index_to_remove;
    $form_state->set('rows_removed', $rows_removed);

    // Update the list of final results.
    $assignments = $form_state->get('assignments');
    unset($assignments[$index_to_remove]);
    $form_state->set('assignments', $assignments);

    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $submitted_data = [];
    foreach ((array) $form_state->get('assignments') as $row) {
      if ($uuid_submitted = $row['category_uuid']) {
        $uuid_pattern = '/[0-9a-fA-F]{8}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{4}-[0-9a-fA-F]{12}/i';
        $matches = [];
        if (preg_match($uuid_pattern, $uuid_submitted, $matches) !== 1) {
          // @todo Handle the (possibly) rare case when submitted value is invalid.
        }

        $submitted_data[$uuid_submitted] = $row['media_bundles'] ?? [];
      }
    }

    // Store received input in module config.
    $this->configFactory->getEditable('acquiadam_asset_import.settings')->set('categories', $submitted_data)->save();

    $media_storage = $this->entityTypeManager->getStorage('media');

    // Start processing and queueing media item creation.
    $queued_count = 0;
    foreach ($submitted_data as $category_uuid => $bundles) {
      try {
        $response = $this->userClientFactory->getUserClient()->getAssetsInCategory($category_uuid);
      }
      catch (\Exception $exception) {
        $this->logger->warning('Unable to fetch assets of a category from Widen API. Error: %message', [
          '%message' => $exception->getMessage(),
        ]);
      }

      // If there is no asset in the category move forward.
      if (!isset($response['items']) || count($response['items']) < 1 || $response['total_count'] === 0) {
        continue;
      }

      // Start iterate through the assets in the given category.
      foreach ($response['items'] as $asset_data) {
        // Get prepared to any case Widen might exposing unavailable content.
        if (!$asset_data['released_and_not_expired']) {
          continue;
        }

        $asset_media_type = $this->mediaTypeResolver->resolve($asset_data);
        // Assets that don't resolve to any media type should be skipped.
        if ($asset_media_type == NULL) {
          continue;
        }
        $asset_media_type_id = $asset_media_type->id();

        // Check whether if its media type is allowed by the filtering.
        if (!empty($bundles) && !in_array($asset_media_type_id, $bundles)) {
          continue;
        }

        $existing_media_entities = $media_storage->loadByProperties([$this->sourceFieldName => $asset_data['id']]);

        // Check whether if the media entity already exists.
        // @todo this duplication checker logic should be centralized. See also: AssetImporter::processItem().
        if (count($existing_media_entities) > 1) {
          // @todo Consider this instead: throw new IntegrityConstraintViolationException('message', 500);.
          $this->logger->error('Invalid state detected: multiple media items (%media_ids) share the same asset ID: %asset_id. Suggested to delete all affected media items and then create them again.', [
            '%media_ids' => implode(', ', array_keys($existing_media_entities)),
            '%asset_id' => $asset_data['id'],
          ]);
          $this->messenger()->addError($this->t('Invalid state detected: multiple media items share the same asset ID. Please contact the site administrator. See the log for details.'));
          continue;
        }
        elseif (count($existing_media_entities) === 1) {
          // Check whether if the existing media entity is the same version.
          $media_entity = array_values($existing_media_entities)[0];
          assert($media_entity instanceof MediaInterface);
          $source_field = $media_entity->get($this->sourceFieldName);
          if (!$source_field->isEmpty() && $source_field->first()->version_id === $asset_data['version_id']) {
            continue;
          }
        }

        $this->assetImportQueue->createItem([
          'target_bundle' => $asset_media_type_id,
          'file_name' => $asset_data['filename'],
          'asset_uuid' => $asset_data['id'],
          'version_id' => $asset_data['version_id'],
          'queuer_uid' => $this->currentUser->id(),
        ]);
        $queued_count++;
      }
    }

    // Display some informational messages.
    $dam_media_list_url = Url::fromRoute('view.dam_content_overview.page_1')->toString();
    if ($queued_count > 0) {
      $this->messenger->addStatus($this->formatPlural($queued_count,
        '1 asset in total was queued for later importing. Once the scheduled task gets processed, the new media item will appear on the <a href=":url">DAM media list</a>.',
        '@count assets in total were queued for later importing. Once the scheduled tasks get processed, the new media items for each of them will appear on the <a href=":url">DAM media list</a>.',
        [':url' => $dam_media_list_url])
      );
    }
    else {
      $this->messenger->addStatus($this->t('No asset was queued for later importing which can be normal if all assets from these categories have been imported already earlier. Depending on cron runs, some of their corresponding media items should appear on the <a href=":url">DAM media list</a> soon.',
        [':url' => $dam_media_list_url])
      );
    }
  }

}
