<?php

declare(strict_types=1);

namespace Drupal\acquiadam_asset_import\Plugin\QueueWorker;

use Drupal\acquia_dam\Entity\MediaSourceField;
use Drupal\acquia_dam\Plugin\QueueWorker\AssetQueueWorkerBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaInterface;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Imports Acquia DAM assets.
 *
 * @QueueWorker (
 *   id = "acquia_dam_asset_import",
 *   title = @Translation("Acquia DAM Asset Importer"),
 *   cron = {"time" = 30}
 * )
 */
class AssetImporter extends AssetQueueWorkerBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Entity storage service for media items.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $mediaStorage;

  /**
   * The HTTP client of Guzzle.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Acquia DAM specific logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $media_storage
   *   Entity storage service for media items.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client of Guzzle.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Psr\Log\LoggerInterface $logger
   *   Acquia DAM specific logger service.
   */
  public function __construct(array $configuration, string $plugin_id, array $plugin_definition, EntityStorageInterface $media_storage, ClientInterface $http_client, MessengerInterface $messenger, LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->mediaStorage = $media_storage;
    $this->httpClient = $http_client;
    $this->messenger = $messenger;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')->getStorage('media'),
      $container->get('http_client'),
      $container->get('messenger'),
      $container->get('logger.channel.acquia_dam'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $source_field_name = MediaSourceField::SOURCE_FIELD_NAME;
    $existing_media_entities = $this->mediaStorage->loadByProperties([$source_field_name => $data['asset_uuid']]);

    // Check whether if the media entity already exists.
    // @todo this duplication checker logic should be centralized. See also: BulkImportConfigForm::submitForm().
    if (count($existing_media_entities) > 1) {
      // @todo Maybe throw new IntegrityConstraintViolationException('message', 500);.
      $this->logger->error('Invalid state detected: multiple media items (%media_ids) share the same asset ID: %asset_id. Suggested to delete all affected media items and then create them again.', [
        '%media_ids' => implode(', ', array_keys($existing_media_entities)),
        '%asset_id' => $data['asset_uuid'],
      ]);

      return NULL;
    }
    elseif (count($existing_media_entities) === 1) {
      // Check whether if the existing media entity is the same version.
      $media_entity = array_values($existing_media_entities)[0];
      assert($media_entity instanceof MediaInterface);
      $source_field = $media_entity->get($source_field_name);
      if (!$source_field->isEmpty() && $source_field->first()->version_id === $data['version_id']) {
        $this->logger->info('DAM asset %file_name (asset ID: %asset_id) was skipped during import as its corresponding media item with ID: %media_id already exists.', [
          '%file_name' => $data['file_name'],
          '%asset_id' => $data['asset_uuid'],
          '%media_id' => $media_entity->id(),
          'link' => $media_entity->toLink($this->t('View'))->toString(),
        ]);

        return NULL;
      }
    }

    $media_entity = $this->mediaStorage->create([
      'bundle' => $data['target_bundle'],
      'name' => $data['file_name'],
      'uid' => $data['queuer_uid'],
      MediaSourceField::SOURCE_FIELD_NAME => [
        'asset_id' => $data['asset_uuid'],
      ],
    ]);
    $media_entity->save();

    $this->logger->info('DAM asset %file_name (asset ID: %asset_id) was created as a media item with ID: %media_id.', [
      '%file_name' => $data['file_name'],
      '%asset_id' => $data['asset_uuid'],
      '%media_id' => $media_entity->id(),
      'link' => $media_entity->toLink($this->t('View'))->toString(),
    ]);
  }

}
