<?php

declare(strict_types=1);

namespace Drupal\acquiadam_asset_import\Form;

use Drupal\acquia_dam\Plugin\media\Source\Asset;
use Drupal\Core\Config\Config;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\media\MediaTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Acquia DAM asset import settings.
 */
class BulkImportConfigForm extends ConfigFormBase {

  /**
   * Acquia DAM client on behalf of the current user.
   *
   * @var \Drupal\acquia_dam\Client\AcquiaDamClientFactory
   */
  protected $client;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Acquia DAM logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The current active user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $currentRouteMatch;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->client = $container->get('acquia_dam.client.factory');
    $instance->messenger = $container->get('messenger');
    $instance->logger = $container->get('logger.channel.acquia_dam');
    $instance->currentUser = $container->get('current_user');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->currentRouteMatch = $container->get('current_route_match');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'acquiadam_asset_import_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['acquiadam_asset_import.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $category_options = $this->getCachedOptions($form_state, 'category_options');
    $asset_options = $this->getCachedOptions($form_state, 'asset_options');
    $asset_descriptions = $this->getCachedOptions($form_state, 'asset_descriptions');
    if (!$category_options) {
      if ($category_options === []) {
        $form['content'] = [
          '#markup' => $this->t('No categories were found available for the currently authorized user account on the remote DAM service. Bulk importing media assets requires at least one category to be available to fetch assets from. Please ensure that at least one category exists and is accessible for the current user or <a href=":url">authenticate with a different user account</a>.', [
            ':url' => Url::fromRoute('entity.user.acquia_dam_auth', [
              'user' => $this->currentUser->id(),
            ])->toString(),
          ]),
        ];
      }
      return $form;
    }
    $selected_category = $form_state->getValue('category_uuid', '');
    $selected_data = $this->getSelectedData($form_state);

    // Wrap entire form into single container.
    $form['wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'form-table-wrapper'],
    ];
    $form['wrapper']['category'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Category to import from'),
    ];

    // Category dropdown.
    $form['wrapper']['category']['category_uuid'] = [
      '#type' => 'select',
      '#title' => $this->t('Source category'),
      '#description' => $this->t('List of categories in the remote DAM system available for the authorized user account. Please choose which of them the media assets should be imported from.'),
      '#options' => $category_options,
      '#empty_option' => $this->t('Add category to import'),
      '#empty_value' => 'none',
      '#sort_options' => TRUE,
      '#ajax' => [
        'callback' => '::updateAssetsContainer',
        'wrapper' => 'media-bundles-container',
      ],
    ];

    // The media bundle list container.
    $form['wrapper']['category']['assets'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'media-bundles-container'],
    ];

    if ($selected_category && $selected_category !== "none") {
      $selected_options = $selected_data[$selected_category] ?? [];
      $selected_options = array_values($selected_options);
      $form['wrapper']['category']['assets']['enable_filter'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Filter assets'),
        '#description' => $this->t('Filter assets based on media type.'),
        '#default_value' => $form_state->get('enable_filter') ?? FALSE,
      ];
      $form['wrapper']['category']['assets']['media_bundles'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Import only assets which would be assigned to these media types'),
        '#sort_options' => TRUE,
        '#multiple' => FALSE,
        '#options' => $asset_options,
        '#default_value' => $selected_options,
        '#states' => [
          'visible' => [
            ':input[name="enable_filter"]' => ['checked' => TRUE],
          ],
        ],
        ...$asset_descriptions,
      ];

      $form['wrapper']['category']['assets']['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => isset($selected_data[$selected_category]) ? 'Update' : '+ Add',
        '#submit' => ['::saveCategoryForImport'],
        '#ajax' => [
          'callback' => '::resetFormValues',
          'wrapper' => 'form-table-wrapper',
        ],
      ];
    }

    // Table of selected categories.
    $form['wrapper']['selected_table'] = [
      '#type' => 'table',
      '#header' => [
        'category_info' => $this->t('From these categories…'),
        'media_bundles' => $this->t('…only these type of media'),
        'remove_button' => '',
      ],
      '#attributes' => ['id' => 'selected-categories-table'],
      '#empty' => $this->t('No category has been selected yet.'),
    ];

    $form['wrapper']['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#button_type' => 'primary',
        '#disabled' => !$selected_data && !$this->loadConfig()->get('categories'),
      ],
      'cancel' => [
        '#type' => 'link',
        '#title' => $this->t('Cancel'),
        '#url' => Url::fromRoute($this->currentRouteMatch->getRouteName()),
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ],
    ];

    foreach ($selected_data as $category => $media_bundles) {
      $form['wrapper']['selected_table'][] = [
        'category_info' => ['#markup' => $category_options[$category] ?? ''],
        'media_bundles' => $this->renderMediaBundles($media_bundles, $asset_options),
        'remove_button' => [
          'data' => [
            '#type' => 'submit',
            '#value' => $this->t('Remove'),
            '#name' => 'delete_' . $category,
            '#submit' => ['::removeCategoryFromImport'],
            '#limit_validation_errors' => [
              ['selected_table'],
            ],
            '#ajax' => [
              'callback' => '::resetFormValues',
              'wrapper' => 'form-table-wrapper',
              'disable-refocus' => TRUE,
            ],
            '#attributes' => [
              'class' => ['button--danger', 'js-form-submit', 'use-ajax'],
              'data-disable-refocus' => 'true',
            ],
          ],
        ],
      ];
    }
    return $form;
  }

  /**
   * AJAX callback to load media_bundles and show Add/Update button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function updateAssetsContainer(array &$form, FormStateInterface $form_state): array {
    foreach ($form['wrapper']['category']['assets']['media_bundles']['#options'] as $key => $label) {
      $form['wrapper']['category']['assets']['media_bundles'][$key]['#checked'] = FALSE;
    }
    $selected_data = array_filter($this->getSelectedData($form_state));
    $selected_category = $form_state->getValue('category_uuid') ?? NULL;
    $selected_options = !empty($selected_data[$selected_category]) ? array_values($selected_data[$selected_category]) : [];
    if ($selected_options) {
      foreach ($form['wrapper']['category']['assets']['media_bundles']['#default_value'] as $value) {
        $form['wrapper']['category']['assets']['media_bundles'][$value]['#attributes']['checked'] = 'checked';
      }
    }
    $form['wrapper']['category']['assets']['enable_filter']['#checked'] = $selected_options;
    return $form['wrapper']['category']['assets'];
  }

  /**
   * AJAX callback to reset form values.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function resetFormValues(array &$form, FormStateInterface $form_state): array {
    $form['wrapper']['category']['category_uuid']['#value'] = 'none';
    $form['wrapper']['category']['assets']['enable_filter']['#access'] = FALSE;
    $form['wrapper']['category']['assets']['media_bundles']['#access'] = FALSE;
    return $form['wrapper'];
  }

  /**
   * Renders the media list in table.
   *
   * @param array $media_bundles
   *   The media bundles to render.
   * @param array $asset_options
   *   The asset options.
   */
  private function renderMediaBundles(array $media_bundles, array $asset_options): array {
    if (empty($media_bundles)) {
      return [
        '#markup' => $this->t('All assets (no filtering)'),
      ];
    }
    $items = [];
    foreach ($media_bundles as $subcategory) {
      $items[] = $asset_options[$subcategory];
    }
    return [
      '#theme' => 'item_list',
      '#items' => $items,
    ];
  }

  /**
   * Saves the selected category and media bundles for import.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function saveCategoryForImport(array &$form, FormStateInterface $form_state): void {
    $selected_category = $form_state->getValue('category_uuid');
    $selected_media_bundles = array_filter($form_state->getValue('media_bundles', []));
    $selected_data = $this->getSelectedData($form_state);
    if ($selected_category) {
      $selected_data[$selected_category] = !empty($selected_media_bundles) ? $selected_media_bundles : [];
      $form_state->set('selected_data', $selected_data);
    }
    $form_state->set('enable_filter', FALSE);
    $form_state->setValue('category_uuid', "none");
    $form_state->setRebuild();
  }

  /**
   * Removes a category from the selected data list.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function removeCategoryFromImport(array &$form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();
    if (!empty($triggering_element['#name'])) {
      $category_to_delete = str_replace('delete_', '', $triggering_element['#name']);
      $selected_data = $this->getSelectedData($form_state);

      // Remove the category from stored data.
      if (isset($selected_data[$category_to_delete])) {
        unset($selected_data[$category_to_delete]);
        $form_state->set('selected_data', $selected_data);
      }
    }
    $form_state->setRebuild();
    return $form['wrapper']['selected_table'];
  }

  /**
   * Returns the selected data from the form state.
   */
  private function getSelectedData($form_state): array {
    if (!$form_state->has('selected_data')) {
      $form_state->set('selected_data', $this->loadConfig()->get('categories') ?? []);
    }
    return $form_state->get('selected_data');
  }

  /**
   * Returns the config import settings configuration object.
   */
  private function loadConfig(): Config {
    return $this->config($this->getEditableConfigNames()[0]);
  }

  /**
   * Fetches categories from the remote DAM service.
   */
  private function fetchCategories(): ?array {
    try {
      $response = $this->client->getUserClient()->getCategories();
    }
    catch (\Exception $exception) {
      $this->messenger->addWarning('Something went wrong while gathering categories from the remote DAM service. Please contact the site administrator.');
      $this->logger->error($exception->getMessage());
      return NULL;
    }
    if (empty($response['total_count'])) {
      return [];
    }
    return isset($response['items']) ? array_column($response['items'], 'name', 'id') : [];
  }

  /**
   * Returns the list of acquia dam asset.
   *
   * @param bool $needs_description
   *   Whether to include descriptions in the options.
   */
  private function getAssetOptions(bool $needs_description = FALSE): array {
    $media_types = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    $media_types = array_filter($media_types, static fn(MediaTypeInterface $media_type) => $media_type->getSource() instanceof Asset);
    return array_map(fn($obj) => ($needs_description ? ['#description' => $obj->get('description')] : $obj->label()), $media_types);
  }

  /**
   * Returns the cached options if available, otherwise fetches & cache them.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Given form state.
   * @param string $key
   *   Given cache key.
   */
  private function getCachedOptions(FormStateInterface $form_state, string $key): ?array {
    if (!$form_state->has($key)) {
      match ($key) {
        'category_options' => $data = $this->fetchCategories(),
        'asset_options' => $data = $this->getAssetOptions(),
        'asset_descriptions' => $data = $this->getAssetOptions(TRUE),
      };
      if (isset($data)) {
        $form_state->set($key, $data);
      }
    }
    return $form_state->get($key);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->loadConfig()
      ->set('categories', $form_state->get('selected_data') ?? [])
      ->save();
    $this->messenger->addStatus($this->t('The configuration settings have been successfully saved.'));
  }

}
