<?php

namespace Drupal\acquiadam_asset_import\Services;

use Drupal\acquia_dam\Client\AcquiaDamClientFactory;
use Drupal\acquia_dam\Entity\MediaSourceField;
use Drupal\acquia_dam\MediaTypeResolver;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\DatabaseQueue;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * A class to add items to acquia_dam_asset_import queue.
 */
class AssetQueueService implements AssetQueueInterface {

  use StringTranslationTrait;

  /**
   * Acquia DAM client on behalf of the current user.
   *
   * @var \Drupal\acquia_dam\Client\AcquiaDamClientFactory
   */
  protected $userClientFactory;

  /**
   * The asset import queue.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected $assetImportQueue;

  /**
   * Configuration.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $config;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Logger channel for Acquia DAM.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $damLoggerChannel;

  /**
   * The current active user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Acquia DAM media type resolver.
   *
   * @var \Drupal\acquia_dam\MediaTypeResolver
   */
  protected $mediaTypeResolver;

  /**
   * Store module-wide used constant as class attribute.
   *
   * @var string
   */
  protected $sourceFieldName;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  private $database;

  /**
   * Constructs an AssetQueueService service object.
   *
   * @param \Drupal\acquia_dam\Client\AcquiaDamClientFactory $user_client_factory
   *   The Acquia DAM client factory.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\Core\Config\ConfigFactory $config
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger_channel
   *   Logger channel instance.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current active user.
   * @param \Drupal\acquia_dam\MediaTypeResolver $media_type_resolver
   *   Acquia DAM media type resolver.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   */
  public function __construct(
    AcquiaDamClientFactory $user_client_factory,
    QueueFactory $queue_factory,
    ConfigFactory $config,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelInterface $logger_channel,
    AccountInterface $current_user,
    MediaTypeResolver $media_type_resolver,
    Connection $connection,
  ) {
    $this->userClientFactory = $user_client_factory;
    $this->assetImportQueue = $queue_factory;
    $this->config = $config;
    $this->entityTypeManager = $entity_type_manager;
    $this->damLoggerChannel = $logger_channel;
    $this->currentUser = $current_user;
    $this->mediaTypeResolver = $media_type_resolver;
    $this->sourceFieldName = MediaSourceField::SOURCE_FIELD_NAME;
    $this->database = $connection;
  }

  /**
   * {@inheritdoc}
   */
  public function addAssetsToQueue(): ?int {
    // Get a list of category names from config.
    $categories = $this->config->get('acquiadam_asset_import.settings')->get('categories');

    // If no categories are configured, return.
    if (empty($categories)) {
      return NULL;
    }

    $asset_import_queue = $this->assetImportQueue->get('acquia_dam_asset_import');

    $items = 0;
    // Start processing and queueing media item creation.
    foreach ($categories as $category_uuid => $bundles) {
      $response = $this->fetchAssetsInCategory($category_uuid);
      if (empty($response)) {
        continue;
      }

      $asset_ids = array_column($response['items'], 'id');
      $non_imported_assets = $this->filterAssetsToImport($asset_ids);

      $filtered_assets = array_filter(array_map(function ($asset_data) use ($bundles, $non_imported_assets) {
        return $this->filterAndFormatAsset($asset_data, $bundles, $non_imported_assets);
      }, $response['items']));

      array_walk($filtered_assets, [$asset_import_queue, 'createItem']);
      $items += count($filtered_assets);
    }

    return $items;
  }

  /**
   * Fetch assets in a category.
   *
   * @param string $category_uuid
   *   The category UUID.
   *
   * @return array|null
   *   The response array or NULL on failure.
   */
  private function fetchAssetsInCategory(string $category_uuid): ?array {
    try {
      $response = $this->userClientFactory->getSiteClient()->getAssetsInCategory($category_uuid);
      if (empty($response['items']) || $response['total_count'] === 0) {
        return NULL;
      }
      return $response;
    }
    catch (\Exception $exception) {
      $this->damLoggerChannel->warning('Unable to fetch assets of a category from Widen API. Error: %message', [
        '%message' => $exception->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Filter and format asset data for queuing.
   *
   * @param array $asset_data
   *   An array of asset data.
   * @param array $bundles
   *   An array of bundles.
   * @param array|null $non_existing_assets
   *   An array of non-existing assets, NULL if otherwise.
   */
  private function filterAndFormatAsset(array $asset_data, array $bundles, ?array $non_existing_assets): ?array {
    if (!$asset_data['released_and_not_expired']) {
      return NULL;
    }

    if ($non_existing_assets && !in_array($asset_data['id'], $non_existing_assets)) {
      return NULL;
    }

    $asset_media_type = $this->mediaTypeResolver->resolve($asset_data);
    if ($asset_media_type === NULL) {
      return NULL;
    }

    $asset_media_type_id = $asset_media_type->id();
    if (!empty($bundles) && !in_array($asset_media_type_id, $bundles)) {
      return NULL;
    }

    // Skip asset if it's already added in the queue.
    $queued_assets = $this->getQueuedAssets();
    if (in_array($asset_data['id'], $queued_assets, TRUE)) {
      return NULL;
    }

    // Return asset data for queuing.
    return [
      'target_bundle' => $asset_media_type_id,
      'file_name' => $asset_data['filename'],
      'asset_uuid' => $asset_data['id'],
      'version_id' => $asset_data['version_id'],
      'queuer_uid' => $this->currentUser->id(),
    ];
  }

  /**
   * Get uuids of already queued assets to prevent duplicates.
   */
  private function getQueuedAssets(): array {
    $items = [];
    try {
      // Fetch queued items.
      $result = $this->database->select(DatabaseQueue::TABLE_NAME, 'q')
        ->fields('q', ['data'])
        ->where('name = :name', [':name' => 'acquia_dam_asset_import'])
        ->execute()
        ->fetchCol();
    }
    catch (\Exception $e) {
      $this->damLoggerChannel->error('Failed to fetch queued items from the database. Error: %error', [
        '%error' => $e->getMessage(),
      ]);
      return $items;
    }
    $items = array_column(array_map(function ($serialized_data) {
      return unserialize($serialized_data, ['allowed_classes' => FALSE]);
    }, $result), 'asset_uuid');
    return array_filter($items);
  }

  /**
   * Returns the list of assets that are not already imported.
   *
   * @param array $asset_ids
   *   An array of asset uuids.
   */
  protected function filterAssetsToImport(array $asset_ids): ?array {
    try {
      // Query the database to get existing assets.
      $query = $this->database->select("media__" . $this->sourceFieldName, 'm')
        ->fields('m', ['acquia_dam_asset_id_asset_id'])
        ->condition('acquia_dam_asset_id_asset_id', $asset_ids, 'IN');
      $existing_uuids = $query->execute()->fetchCol();
    }
    catch (\Exception $e) {
      $this->damLoggerChannel->error('Failed to fetch existing assets from the database. Error: %error', [
        '%error' => $e->getMessage(),
      ]);
      return NULL;
    }

    // Return UUIDs that do not exist in the database.
    return array_diff($asset_ids, $existing_uuids);
  }

}
