<?php

declare(strict_types=1);

namespace Drupal\acquia_dam\Plugin\media\acquia_dam;

use Drupal\acquia_dam\Plugin\media\Source\Asset;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Manages DAM Asset Media Source plugin definitions.
 *
 * Each media source definition array is set in the dervier's annotation.
 */
final class AssetMediaSourceManager extends DefaultPluginManager {

  /**
   * Constructs a new DAM Asset Media Source plugin manager.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations,.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct('Plugin/media/acquia_dam', $namespaces, $module_handler, 'Drupal\acquia_dam\Plugin\media\acquia_dam\MediaSourceTypeInterface', 'Drupal\acquia_dam\Annotation\AssetMediaSource');

    $this->setCacheBackend($cache_backend, 'acquia_dam_media_source_plugins');
    $this->alterInfo('acquia_dam_media_source');
  }

  /**
   * {@inheritdoc}
   */
  public function getDefinitions() {
    $definitions = parent::getDefinitions();
    foreach ($definitions as &$definition) {
      $definition['asset_class'] = $definition['class'];
      $definition['class'] = Asset::class;
    }
    return $definitions;
  }

  /**
   * Helper function to swap asset fields for a provided view display.
   *
   * @param \Drupal\Core\Config\Config|\Drupal\Core\Entity\Display\EntityViewDisplayInterface $view_display
   *   Current view display to swap fields.
   * @param string $existing_field_name
   *   Existing field to disable in the display.
   * @param string $active_field_name
   *   Name of the (newly) active field.
   * @param array $active_field
   *   Settings for the (newly) active field.
   *
   * @return Config|EntityViewDisplay
   *   This method can be used for both Config objects and Entity View Displays.
   */
  public static function swapAssetFields($view_display, string $existing_field_name, string $active_field_name, array $active_field) {
    // Update the hidden fields.
    $hidden_region = array_filter($view_display->get('hidden'), fn($field_name) => $field_name !== $active_field_name, ARRAY_FILTER_USE_KEY);
    $hidden_region[$existing_field_name] = true;

    // Update field label & weight.
    $active_field['label'] = $content_region[$existing_field_name]['label'] ?? 'hidden';
    $active_field['weight'] = $content_region[$existing_field_name]['weight'] ?? 0;
    $content_region[$active_field_name] = $active_field;
    // Remove inactive fields from content region.
    unset($content_region[$existing_field_name]);

    // Update view mode configuration.
    $view_display->set('content', $content_region);
    $view_display->set('hidden', $hidden_region);
    return $view_display;
  }

}
